package dynamodb

import (
	"fmt"

	"code.justin.tv/common/ddbmetrics"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"
)

type rowProcessor func(map[string]*dynamodb.AttributeValue) error

// ChunkedBatchGet performs a batch get against a single table.  Requests are split into
// chunks and sent serially to dynamodb as individual batch get requests.
func ChunkedBatchGet(
	db dynamodbiface.DynamoDBAPI,
	metrics *ddbmetrics.Publisher,
	tableName string,
	keys []map[string]*dynamodb.AttributeValue,
	accumulate rowProcessor) error {

	const maxBatchLimit int = 100

	for start := 0; start < len(keys); start += maxBatchLimit {
		end := start + maxBatchLimit
		if end > len(keys) {
			end = len(keys)
		}

		input := &dynamodb.BatchGetItemInput{
			RequestItems: map[string]*dynamodb.KeysAndAttributes{
				tableName: {
					Keys:           keys[start:end],
					ConsistentRead: aws.Bool(false),
				},
			},
			ReturnConsumedCapacity: aws.String("INDEXES"),
		}

		var out *dynamodb.BatchGetItemOutput
		out, err := db.BatchGetItem(input)
		if err != nil {
			return fmt.Errorf("Error from BatchGetItem: %v", err)
		}

		if metrics != nil {
			for _, consumed := range out.ConsumedCapacity {
				metrics.Report(ddbmetrics.Write, consumed)
			}
		}

		for _, item := range out.Responses[tableName] {
			err = accumulate(item)

			if err != nil {
				return fmt.Errorf("Error accumulating row: %v", err)
			}
		}
	}
	return nil
}

// ChunkedBatchQuery performs a query against a single table, continuing the
// query across multiple segments if neccessary.
func ChunkedBatchQuery(
	db dynamodbiface.DynamoDBAPI,
	metrics *ddbmetrics.Publisher,
	input *dynamodb.QueryInput,
) ([]*dynamodb.QueryOutput, error) {
	out := []*dynamodb.QueryOutput{}
	working := true
	for working {
		segment, err := db.Query(input)
		if err != nil {
			return nil, err
		}
		out = append(out, segment)
		if segment.LastEvaluatedKey != nil {
			input.ExclusiveStartKey = segment.LastEvaluatedKey
		} else {
			working = false
		}
	}
	return out, nil
}

// ChunkedBatchWrite performs a batch write against a single table.  Requests
// are split into chunks and sent serially to dynamodb as individual batch write
// requests.
func ChunkedBatchWrite(
	db dynamodbiface.DynamoDBAPI,
	metrics *ddbmetrics.Publisher,
	tableName string,
	requests []*dynamodb.WriteRequest) error {

	const maxBatchLimit int = 25

	// maintain a queue here in case we need to retry a failed delete

	requestQueue := []*dynamodb.WriteRequest{}
	for _, request := range requests {
		requestQueue = append(requestQueue, request)
	}

	for start := 0; start < len(requestQueue); start += maxBatchLimit {
		end := start + maxBatchLimit
		if end > len(requestQueue) {
			end = len(requestQueue)
		}

		input := &dynamodb.BatchWriteItemInput{
			RequestItems: map[string][]*dynamodb.WriteRequest{
				tableName: requestQueue[start:end],
			},
			ReturnConsumedCapacity: aws.String("INDEXES"),
		}

		var out *dynamodb.BatchWriteItemOutput
		out, err := db.BatchWriteItem(input)
		if err != nil {
			return translateDynamoError(err)
		}
		if metrics != nil {
			for _, consumed := range out.ConsumedCapacity {
				metrics.Report(ddbmetrics.Write, consumed)
			}
		}

		for _, item := range out.UnprocessedItems[tableName] {
			// Anything that failed, just put back on the queue.
			requestQueue = append(requestQueue, item)
		}
	}
	return nil
}
