package dynamodb

import (
	"errors"
	"fmt"
	"sync"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"
)

// ClearTable wipes a dynamo table using N background threads to consume pages
// of rows produced by the foreground. Recommend values for N are 1-3; higher
// values are faster at the cost of additional write IOPs.
func ClearTable(db dynamodbiface.DynamoDBAPI, tableName string, threads int) error {
	if threads < 1 {
		return errors.New("Illegal thread count specified")
	}
	description, err := db.DescribeTable(&dynamodb.DescribeTableInput{TableName: aws.String(tableName)})
	if err != nil {
		return err
	}

	keyAttributes := []string{}
	expressionAttributeNames := make(map[string]*string)
	projection := ""

	for index, schema := range description.Table.KeySchema {
		alias := fmt.Sprintf("#key%d", index)
		expressionAttributeNames[alias] = schema.AttributeName
		keyAttributes = append(keyAttributes, *schema.AttributeName)
		if len(projection) > 0 {
			projection = projection + ","
		}
		projection = projection + alias
	}

	batches := make(chan []*dynamodb.WriteRequest)
	errors := make(chan error, threads)
	var active sync.WaitGroup
	active.Add(threads)

	// consume pages and delete in the background
	for i := 0; i < threads; i++ {
		go func() {
			defer active.Done()
			for deletions := range batches {
				if chErr := ChunkedBatchWrite(db, nil, tableName, deletions); chErr != nil {
					errors <- chErr
					break
				}
			}
		}()
	}

	var deleteErr error
	err = db.ScanPages(&dynamodb.ScanInput{
		TableName:                aws.String(tableName),
		ProjectionExpression:     aws.String(projection),
		ExpressionAttributeNames: expressionAttributeNames,
	}, func(page *dynamodb.ScanOutput, lastPage bool) bool {
		deletions := []*dynamodb.WriteRequest{}
		for _, item := range page.Items {
			key := map[string]*dynamodb.AttributeValue{}
			for _, attr := range keyAttributes {
				key[attr] = item[attr]
			}
			request := new(dynamodb.WriteRequest).SetDeleteRequest(new(dynamodb.DeleteRequest).SetKey(key))
			deletions = append(deletions, request)
		}
		batches <- deletions // this will block when there is already 1 queued batch so we don't read too far ahead
		select {
		case deleteErr = <-errors:
			return false
		default:
			return true
		}
	})
	close(batches) // end background thread

	active.Wait() // spin down all threads
	close(errors)
	for deleteErr = range errors { // check final answer
	}
	if err == nil {
		err = deleteErr
	}
	return err
}
