package lazy

import (
	"errors"
	"reflect"

	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
)

type AttributeMap map[string]*dynamodb.AttributeValue
type MigrationFunction func(AttributeMap) error
type MigrationMap map[int]MigrationFunction

const lazyMigrationMarker = "migration_format_version"

var (
	ErrMigrationBadVersion = errors.New("Unable to migrate; version handler missing")
	ErrCodeOutOfDate       = errors.New("Unable to migrate; code is out of date")
	ErrCouldNotAnnotate    = errors.New("Unable to migrate; value must be a struct with no `migration_format_version` field")
	ErrMigrationNotFound   = errors.New("Unable to migrate; annotation not found")
	ErrBackwardUnsupported = errors.New("Unable to migrate; backward version unsupported")
	ErrMigrationCorrupt    = errors.New("Unable to migrate; data was not in expected format")
)

// Marshal wraps the dynamodb marshal call to enable lazy migration
func Marshal(in interface{}) (*dynamodb.AttributeValue, error) {
	if comp, ok := in.(Migratable); ok {
		return dynamodbattribute.Marshal(Wrap(comp))
	}
	return dynamodbattribute.Marshal(in)
}

// Unmarshal wraps the dynamodb unmarshal call to enable lazy migration
func Unmarshal(av *dynamodb.AttributeValue, out interface{}) error {
	if comp, ok := out.(Migratable); ok && !reflect.ValueOf(out).IsNil() {
		return dynamodbattribute.Unmarshal(av, Wrap(comp))
	}
	return dynamodbattribute.Unmarshal(av, out)
}

// UnmarshalMap wraps a map of field names to attribute values as a dynamo
// map attribute for convenience
func UnmarshalMap(m map[string]*dynamodb.AttributeValue, out interface{}) error {
	return Unmarshal(&dynamodb.AttributeValue{M: m}, out)
}

// MarshalMap wraps a map of field names to attribute values as a dynamo
// map attribute for convenience
func MarshalMap(in interface{}) (map[string]*dynamodb.AttributeValue, error) {
	av, err := Marshal(in)
	if err != nil || av == nil || av.M == nil {
		return map[string]*dynamodb.AttributeValue{}, err
	}

	return av.M, nil
}
