package lazy

import (
	"reflect"

	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
)

type encoder interface {
	Encode(in interface{}) (*dynamodb.AttributeValue, error)
}

type decoder interface {
	Decode(av *dynamodb.AttributeValue, out interface{}, opts ...func(*dynamodbattribute.Decoder)) error
}

// Wrapper is an indirection which is stupid and gross but required
// because the dynamodbattribute system doesn't allow turning off the marshaler
// override check and exports none of its implementation. When making a format
// change, use the Wrap function to upgrade the raw attributes of legacy records
// by making whatever changes are necessary for compitability before it is
// passed to the decoder. This struct takes an encoder/decoder pair injection
// for test override purposes.
type Wrapper struct {
	content Migratable
	encoder encoder
	decoder decoder
}

// Wrap returns the wrapper necessary to (de)serialize the versioned item
// safely to dynamo.
func Wrap(comp Migratable) *Wrapper {
	return &Wrapper{comp, dynamodbattribute.NewEncoder(), dynamodbattribute.NewDecoder()}
}

// Unwrap extracts the original value from the wrapper
func (w *Wrapper) Unwrap() Migratable {
	return w.content
}

// MarshalDynamoDBAttributeValue implements dynamodbattribute.Marshaler
func (w *Wrapper) MarshalDynamoDBAttributeValue(av *dynamodb.AttributeValue) error {
	var err error
	var content interface{} = w.content
	if complex, ok := w.content.(MigratableWithMemento); ok {
		content, err = complex.GetMemento()
		if err != nil {
			return err
		}
	}
	internal, err := w.encoder.Encode(content)
	if err != nil {
		return err
	}
	// convert not-nil but empty instance so we can record our version information
	if internal.NULL != nil {
		if reflect.ValueOf(content).IsNil() {
			av.SetNULL(true)
			return nil
		}
		internal.M = make(AttributeMap)
	}
	targetVersion := w.content.StoreAsVersion()
	if err = w.migrate(w.content.FormatVersion(), targetVersion, internal.M); err != nil {
		return err
	}
	av.SetM(internal.M)
	if av.M == nil || av.M[lazyMigrationMarker] != nil {
		return ErrCouldNotAnnotate
	}
	if targetVersion > 0 {
		av.M[lazyMigrationMarker], _ = w.encoder.Encode(targetVersion)
	}
	return nil
}

// UnmarshalDynamoDBAttributeValue implements dynamodbattribute.Unmarshaler
func (w *Wrapper) UnmarshalDynamoDBAttributeValue(av *dynamodb.AttributeValue) error {
	if av.NULL != nil {
		w.decoder.Decode(av, w.content) // preserve base functionality -- zero all values
		return nil
	}
	if av.M == nil {
		return ErrMigrationNotFound // we only version structs/maps
	}
	var err error
	var format int
	cloned := false
	if av.M[lazyMigrationMarker] != nil {
		av = cloneAttribute(av)
		cloned = true
		if err = w.decoder.Decode(av.M[lazyMigrationMarker], &format); err != nil {
			return err
		}
		delete(av.M, lazyMigrationMarker)
	}
	if format != w.content.FormatVersion() {
		if !cloned {
			av = cloneAttribute(av)
			cloned = true
		}
		if err = w.migrate(format, w.content.FormatVersion(), av.M); err != nil {
			return err
		}
	}
	var content interface{} = w.content
	complex, hasMemento := w.content.(MigratableWithMemento)
	if hasMemento {
		content, err = complex.GetMemento()
		if err != nil {
			return err
		}
	}
	if err = w.decoder.Decode(av, content); err != nil {
		return err
	}
	if hasMemento {
		return complex.ApplyMemento(content)
	}
	return nil
}

func (w *Wrapper) migrate(from, to int, data AttributeMap) error {
	if from > w.content.FormatVersion() {
		return ErrCodeOutOfDate
	}
	forward, backward := w.content.Migrations()
	for ; from < to; from++ {
		f := forward[from]
		if f == nil {
			return ErrMigrationBadVersion
		}
		if err := f(data); err != nil {
			return err
		}
	}
	for ; from > to; from-- {
		f := backward[from]
		if f == nil {
			return ErrBackwardUnsupported
		}
		if err := f(data); err != nil {
			return err
		}
	}
	return nil
}
