package event

import (
	"fmt"
	"sync"
)

type MessageGenerator func(Change, int, []byte) (Message, error)
type byFormat map[int]MessageGenerator

// MessageFactory allows registration and generation of message types from serialized form
type MessageFactory struct {
	gens  map[string]byFormat
	mutex sync.RWMutex
}

func NewMessageFactory() *MessageFactory {
	return &MessageFactory{gens: make(map[string]byFormat)}
}

func (m *MessageFactory) Register(topic Topic, change Change, formatVersions []int, gen MessageGenerator) {
	defer m.mutex.Unlock()
	m.mutex.Lock()

	var ok bool
	var fmts byFormat
	key := getKey(topic, change)
	if fmts, ok = m.gens[key]; !ok {
		fmts = make(byFormat)
		m.gens[key] = fmts
	}
	for _, v := range formatVersions {
		fmts[v] = gen
	}
}

func (m *MessageFactory) Create(topic Topic, change Change, formatVersion int, body []byte) (Message, error) {
	defer m.mutex.RUnlock()
	m.mutex.RLock()

	if fmts, ok := m.gens[getKey(topic, change)]; ok {
		if gen, ok := fmts[formatVersion]; ok {
			return gen(change, formatVersion, body)
		}
		return nil, UnsupportedFormatVersionError{topic, change, formatVersion}
	}
	return nil, UnregisteredMessageTypeError{topic, change}
}

func getKey(topic Topic, change Change) string {
	return fmt.Sprintf("%v/%v", topic, change)
}
