package sns

import (
	"errors"
	"strconv"

	"code.justin.tv/gds/gds/golibs/event"
)

var (
	ErrRecordWasNil   = errors.New("The SNS message record was nil")
	ErrMissingTopic   = errors.New("The SNS message didn't have a topic set")
	ErrMissingChange  = errors.New("The SNS message didn't have a change type set")
	ErrMissingVersion = errors.New("The SNS message didn't have a format version set")
	ErrInvalidVersion = errors.New("The SNS message format version wasn't an integer")
)

type Parser interface {
	Parse(record *Record) (event.Message, error)
}

type parser struct {
	factory *event.MessageFactory
}

func NewParser(factory *event.MessageFactory) Parser {
	return &parser{factory}
}

func (p *parser) Parse(record *Record) (event.Message, error) {
	if record == nil {
		return nil, ErrRecordWasNil
	}
	topic, err := parseTopic(record)
	if err != nil {
		return nil, err
	}
	change, err := parseChange(record)
	if err != nil {
		return nil, err
	}
	fmt, err := parseFormatVersion(record)
	if err != nil {
		return nil, err
	}
	// TODO : convert lambdas to use this lib so we can encode message bodies
	return p.factory.Create(topic, change, fmt, []byte(record.Sns.Message))
}

func parseTopic(record *Record) (event.Topic, error) {
	if record.Sns.Subject == "" {
		return event.Topic(""), ErrMissingTopic
	}
	return event.Topic(record.Sns.Subject), nil
}

func parseChange(record *Record) (event.Change, error) {
	out, ok := record.Sns.MessageAttributes[eventAttrChange]
	if !ok || out.Value == "" {
		return event.Change(""), ErrMissingChange
	}
	return event.Change(out.Value), nil
}

func parseFormatVersion(record *Record) (int, error) {
	fmt, ok := record.Sns.MessageAttributes[eventAttrFormatVersion]
	if !ok {
		return 0, ErrMissingVersion
	}
	fmt64, err := strconv.ParseInt(fmt.Value, 10, 64)
	if err != nil {
		return 0, ErrInvalidVersion
	}
	return int(fmt64), nil
}
