package sns

import (
	"encoding/json"
	"strconv"

	"code.justin.tv/gds/gds/golibs/event"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/sns"
	"github.com/aws/aws-sdk-go/service/sns/snsiface"
)

const (
	eventAttrChange        = "change"
	eventAttrFormatVersion = "format_version"
)

type sender struct {
	sns      snsiface.SNSAPI
	topicARN *string
	attr     map[string]*sns.MessageAttributeValue
}

// NewSender constructs a pipe to SNS that delivers the base64 encoded message body as its Message
func NewSender(svc snsiface.SNSAPI, topicARN string, attr map[string]string) event.Sender {
	return &sender{svc, aws.String(topicARN), convert(attr)}
}

func (s *sender) Send(msg event.Message) error {
	payload, err := build(msg, s.topicARN, s.attr)
	if err != nil {
		return err
	}
	_, err = s.sns.Publish(payload)
	return err
}

func convert(kv map[string]string) map[string]*sns.MessageAttributeValue {
	attr := make(map[string]*sns.MessageAttributeValue)
	for k, v := range kv {
		attr[k] = new(sns.MessageAttributeValue).SetDataType("String").SetStringValue(v)
	}
	return attr
}

func build(msg event.Message, arn *string, in map[string]*sns.MessageAttributeValue) (*sns.PublishInput, error) {
	// TODO : convert lambdas to use this lib so we can encode message bodies
	bytes, err := json.Marshal(msg.Body())
	if err != nil {
		return nil, err
	}

	attrs := make(map[string]*sns.MessageAttributeValue)
	attrs[eventAttrFormatVersion] = new(sns.MessageAttributeValue).SetDataType("Number").SetStringValue(strconv.FormatInt(int64(msg.FormatVersion()), 10))
	attrs[eventAttrChange] = new(sns.MessageAttributeValue).SetDataType("String").SetStringValue(string(msg.Change()))
	for k, v := range in {
		attrs[k] = v
	}
	return &sns.PublishInput{
		TopicArn:          arn,
		Subject:           aws.String(string(msg.Topic())),
		Message:           aws.String(string(bytes)),
		MessageAttributes: attrs,
	}, nil
}
