package http

import (
	"context"
	"io"
	"net/http"
	"strconv"

	"code.justin.tv/gds/gds/golibs/errors"
	"code.justin.tv/gds/gds/golibs/errors/errconv"
	"code.justin.tv/gds/gds/golibs/hystrix"
	"code.justin.tv/gds/gds/golibs/metrics"
	"code.justin.tv/gds/gds/golibs/params"
	"github.com/jixwanwang/apiutils"
)

// APICall defines a generic interface for functions to be plugged into an HTTP
// service.
type APICall func(context.Context) (interface{}, error)

// APIHandler wraps an APICall for use with a Mux; it implements
// metrics.InstrumentedHandler and hystrix.NamedHandler
type APIHandler struct {
	call APICall
	name string
	freq metrics.Frequency
}

var _ hystrix.NamedHandler = new(APIHandler)
var _ metrics.InstrumentedHandler = new(APIHandler)

// NewAPIHandler creates an adapter around an API call and registers a hystrix
// circuit for the handler.
func NewAPIHandler(name string, freq metrics.Frequency, call APICall) http.Handler {
	return &APIHandler{call, name, freq}
}

func (h *APIHandler) Name() string                 { return h.name }
func (h *APIHandler) Frequency() metrics.Frequency { return h.freq }
func (h *APIHandler) ServeHTTP(w http.ResponseWriter, r *http.Request) {
	defer tryClose(r.Body)
	ctx := params.ExtractFromHTTP(r.Context(), r)
	ctx = withPassThroughHack(ctx, r) // HACK: for use with DeprecatedCall
	out, rawErr := h.call(ctx)
	if rawErr != nil {
		err := errconv.LoadConverter(ctx).Convert(rawErr)
		code := errors.GetHTTPStatusOrDefault(err, 500)
		// Serve 2XX responses (201, 206, etc.)
		if code >= 200 && code < 300 {
			w.WriteHeader(code)
			apiutils.ServeJSON(w, out)
		} else {
			body := errors.MarshalErrorForHTTP(err)
			w.Header().Set("Content-Length", strconv.Itoa(len(body)))
			w.Header().Set("Content-Type", "application/json")
			w.WriteHeader(code)
			w.Write(body)
		}
		return
	}

	if out == nil {
		w.WriteHeader(http.StatusNoContent)
		return
	}
	apiutils.ServeJSON(w, out)
}

func tryClose(body io.Closer) {
	if body != nil {
		_ = body.Close()
	}
}
