package metrics

import (
	"errors"
	"log"
	"sync"
	"time"

	"code.justin.tv/gds/gds/golibs/infra"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/cloudwatch"
	"github.com/aws/aws-sdk-go/service/cloudwatch/cloudwatchiface"
	"github.com/mitchellh/hashstructure"
	"github.com/rcrowley/go-metrics"
)

// Unlike many libraries we infer a lot from the environment and do not provide
// handles to control those. As such, and to allow testing we expose these variables
// so that tests can set the values
var client cloudwatchiface.CloudWatchAPI
var reportingInterval = 1 * time.Minute
var defaultDimensions = map[string]string{}
var namespace = ""
var awsConfig *aws.Config

// nowFunc in non-test environments is `time.Now()`, set by `setNowFunc()`
var nowFunc nowFuncT

type nowFuncT func() time.Time

var statters = &instanceHolder{
	instances: map[uint64]*DimensionRegistry{},
}

type instanceHolder struct {
	sync.Mutex
	instances map[uint64]*DimensionRegistry
}

var setupOnce sync.Once

func createSetup(config infra.InfraConfigProvider) func() {
	return func() {
		if err := setNamespace(config); err != nil {
			log.Printf("[WARN] component=cloudwatch-setup fn=setup at=namespace error=%s\n", err)
			return
		}
		setDefaultDimensions(config)
		setNowFunc()

		go tick()

		newClient()
	}
}

// SetAWSConfig sets the AWS config for the client. This must be
// called before any calls to CloudWatch(), otherwise a default config
// is used.
func SetAWSConfig(conf *aws.Config) {
	awsConfig = conf
}

// CloudWatch returns to you a statsd.Statter keyed to the specific dimensions passed in
func CloudWatch(dimensions map[string]string) *DimensionRegistry {
	setupOnce.Do(createSetup(nil))

	h, err := hashstructure.Hash(dimensions, nil)
	if err != nil {
		// keep going, run with only default dimensions
		log.Printf("[WARN] component=cloudwatch-config fn=CloudWatch at=hash error=%s\n", err)
		h = 0
	}

	// protect against a race where the the statter instance for this dimension set gets created twice
	// and loses the first set of data passed to it
	statters.Lock()
	defer statters.Unlock()

	if _, ok := statters.instances[h]; !ok {
		statters.instances[h] = &DimensionRegistry{
			registry:   metrics.NewRegistry(),
			dimensions: dimensions,
		}
	}

	return statters.instances[h]
}

// ConfigurableCloudWatch returns to you a statsd.Statter keyed to the specific dimensions passed in
// using the configuration values from the provided config provider
func ConfigurableCloudWatch(dimensions map[string]string, config infra.InfraConfigProvider) *DimensionRegistry {
	setupOnce.Do(createSetup(config))

	h, err := hashstructure.Hash(dimensions, nil)
	if err != nil {
		// keep going, run with only default dimensions
		log.Printf("[WARN] component=cloudwatch-config fn=CloudWatch at=hash error=%s\n", err)
		h = 0
	}

	// protect against a race where the the statter instance for this dimension set gets created twice
	// and loses the first set of data passed to it
	statters.Lock()
	defer statters.Unlock()

	if _, ok := statters.instances[h]; !ok {
		statters.instances[h] = &DimensionRegistry{
			registry:   metrics.NewRegistry(),
			dimensions: dimensions,
		}
	}

	return statters.instances[h]
}

// Flush sends any local pending metrics immediately
func Flush() {
	emitMetrics()
}

// With returns a DimensionRegistry with the supplied dimensions merged into the existing set
func (d *DimensionRegistry) With(dimensions map[string]string) *DimensionRegistry {
	return CloudWatch(merge(d.dimensions, dimensions))
}

func setDefaultDimensions(config infra.InfraConfigProvider) {
	if config != nil && config.InstanceID() != "" {
		defaultDimensions["instanceId"] = config.InstanceID()
	} else {
		defaultDimensions["instanceId"] = infra.InstanceID()
	}
	if config != nil && config.InstanceID() != "" {
		defaultDimensions["component"] = config.ComponentName()
	} else {
		defaultDimensions["component"] = infra.ComponentName()
	}
	if config != nil && config.InstanceID() != "" {
		defaultDimensions["env"] = config.AppEnv()
	} else {
		defaultDimensions["env"] = infra.AppEnv()
	}
}

func setNamespace(config infra.InfraConfigProvider) error {
	if namespace != "" {
		return nil
	}
	if config != nil && config.AppName() != "" {
		namespace = config.AppName()
		return nil
	}
	if infra.AppName() == "" {
		return errors.New("APP_NAME env var not set")
	}
	namespace = infra.AppName()
	return nil
}

func setNowFunc() {
	if nowFunc != nil {
		return
	}
	nowFunc = time.Now
}

func newClient() {
	if client != nil {
		return
	}

	if awsConfig == nil {
		awsConfig = aws.NewConfig().WithCredentialsChainVerboseErrors(true)
	}
	sess, err := session.NewSession(awsConfig)
	if err != nil {
		log.Printf("[ERROR] component=cloudwatch-config fn=newClient at=error error=%s\n", err)
	}

	client = cloudwatch.New(sess)
}

// merge maps a and b, b getting precendence
func merge(a, b map[string]string) map[string]string {
	m := make(map[string]string)

	for k, v := range a {
		m[k] = v
	}

	for k, v := range b {
		m[k] = v
	}
	return m
}

// CloudwatchConfig is a simple data holder that can be used for the cloudwatch object
// to avoid the use of common.config
type CloudwatchConfig struct {
	appName       string
	appEnv        string
	componentName string
	instanceID    string
}

func NewCloudwatchConfig(appName, appEnv, componentName, instanceID string) *CloudwatchConfig {
	return &CloudwatchConfig{
		appName:       appName,
		appEnv:        appEnv,
		componentName: componentName,
		instanceID:    instanceID,
	}
}

// AppName returns the application name
func (c *CloudwatchConfig) AppName() string {
	return c.appName
}

// AppEnv returns the applications environment
func (c *CloudwatchConfig) AppEnv() string {
	return c.appEnv
}

// ComponentName returns the component name
func (c *CloudwatchConfig) ComponentName() string {
	return c.componentName
}

// InstanceID returns the instance-id if this is on an AWS instance, blank otherwise
func (c *CloudwatchConfig) InstanceID() string {
	return c.instanceID
}
