package metrics

import "sync"

// EventAdapter allows use of an Instrumentor to measure arbitrary code without
// actually wrapping it; useful for situations where the metrics should be sent
// even when a panic ocurs.
type EventAdapter struct {
	code int
	wg   sync.WaitGroup
}

// NewEventAdapter returns a new EventAdapter
func NewEventAdapter() *EventAdapter {
	adapter := &EventAdapter{}
	adapter.wg.Add(1)
	return adapter
}

// Wait should be passed to Instrumentor.Instrument before the code to measure
// is called
func (e *EventAdapter) Wait(args ...interface{}) int {
	e.wg.Wait()
	return e.code
}

// Complete should be called after the code has run; this can be deferred to
// provide a panic-safe accounting of the event
func (e *EventAdapter) Complete(code *int) {
	if code != nil {
		e.code = *code
	}
	e.wg.Done()
}
