package metrics

import "net/http"

type InstrumentedHandler interface {
	http.Handler
	Name() string
	Frequency() Frequency
}

type instrumentedHandler struct {
	call http.Handler
	name string
	freq Frequency
}

func (i *instrumentedHandler) Name() string         { return i.name }
func (i *instrumentedHandler) Frequency() Frequency { return i.freq }
func (i *instrumentedHandler) ServeHTTP(w http.ResponseWriter, r *http.Request) {
	i.call.ServeHTTP(w, r)
}

// InstrumentHandler adopts an http.Handler for use with the Instrument middleware;
// the result is also compatible with the hystrix.CreateCircuit middleware.
func InstrumentHandler(name string, freq Frequency, call http.Handler) http.Handler {
	return &instrumentedHandler{call, name, freq}
}

// InstrumentHandlerFunc adopts an http.HandlerFunc for use with the Instrument middleware;
// the result is also compatible with the hystrix.CreateCircuit middleware.
func InstrumentHandlerFunc(name string, freq Frequency, f func(w http.ResponseWriter, r *http.Request)) http.Handler {
	return &instrumentedHandler{http.HandlerFunc(f), name, freq}
}
