package metrics

import (
	"log"
	"sort"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/cloudwatch"
	"github.com/rcrowley/go-metrics"
)

// much of this comes from https://github.com/sclasen/go-metrics-cloudwatch (MIT)

func tick() {
	ticks := time.NewTicker(reportingInterval)
	defer ticks.Stop()
	for {
		select {
		case <-ticks.C:
			emitMetrics()
		}
	}
}

func emitMetrics() {
	for _, holder := range statters.instances {
		// we want to prevent folks from overriding things like "environment" which come from
		// the defaultDimensions - so let it be the one that overrides
		dimensions := merge(holder.dimensions, defaultDimensions)

		data := metricsData(holder.registry, dimensions)

		for len(data) > 20 {
			put := data[0:20]
			putMetrics(put)
			data = data[20:]
		}

		if len(data) > 0 {
			putMetrics(data)
		}
	}
}

func putMetrics(data []*cloudwatch.MetricDatum) {
	req := &cloudwatch.PutMetricDataInput{
		Namespace:  aws.String(namespace),
		MetricData: data,
	}

	if _, err := client.PutMetricData(req); err != nil {
		log.Printf("[ERROR] component=cloudwatch-sender fn=emitMetrics at=error error=%s\n", err)
	}
}

func metricsData(registry metrics.Registry, dimensions map[string]string) []*cloudwatch.MetricDatum {
	data := []*cloudwatch.MetricDatum{}
	timestamp := aws.Time(nowFunc())

	aDatum := func(name string) *cloudwatch.MetricDatum {
		return &cloudwatch.MetricDatum{
			MetricName: aws.String(name),
			Timestamp:  timestamp,
			Dimensions: mapToDimensions(dimensions),
		}
	}

	registry.Each(func(name string, i interface{}) {
		var datum = aDatum(name)

		switch metric := i.(type) {

		case metrics.Counter:
			datum.Value = aws.Float64(float64(metric.Count()))
			datum.Unit = aws.String(cloudwatch.StandardUnitCount)
			metric.Clear()

		case metrics.Timer:
			t := metric.Snapshot()
			if t.Count() == 0 {
				return
			}

			// We get Nanoseconds back from go-metrics, so we must convert to the right unit
			datum.StatisticValues = &cloudwatch.StatisticSet{
				Maximum:     aws.Float64(float64(time.Duration(t.Max()) / time.Millisecond)),
				Minimum:     aws.Float64(float64(time.Duration(t.Min()) / time.Millisecond)),
				SampleCount: aws.Float64(float64(t.Count())),
				Sum:         aws.Float64(float64(time.Duration(t.Sum()) / time.Millisecond)),
			}
			datum.Unit = aws.String(cloudwatch.StandardUnitMilliseconds)
		}

		data = append(data, datum)
	})

	return data
}

func mapToDimensions(dimensions map[string]string) []*cloudwatch.Dimension {
	ds := []*cloudwatch.Dimension{}
	// enforce the order that we create dimensions for testing purposes
	for _, k := range keySet(dimensions) {
		// trim blank values
		if dimensions[k] == "" {
			continue
		}

		d := &cloudwatch.Dimension{
			Name:  aws.String(k),
			Value: aws.String(dimensions[k]),
		}

		ds = append(ds, d)
	}
	return ds
}

// Dimension should be listed in a stable order, mainly to help with testing
func keySet(in map[string]string) []string {
	keys := []string{}
	for k := range in {
		keys = append(keys, k)
	}
	sort.Strings(keys)

	return keys
}
