package metrics

import (
	"fmt"
	"time"

	"github.com/cactus/go-statsd-client/statsd"
)

type stattingInstrumentor struct {
	stats statsd.StatSender
	cloud *DimensionRegistry
	rates map[Frequency]float32
}

// NewStattingInstrumentor creates an Instrumntor that generates stats based on the included frequencies
func NewStattingInstrumentor(s statsd.StatSender, r map[Frequency]float32) Instrumentor {
	cloud := CloudWatch(nil)
	return &stattingInstrumentor{s, cloud, r}
}

func (s *stattingInstrumentor) Instrument(inner Event, name string, predicted Frequency) Event {
	rate, ok := s.rates[predicted]
	if !ok {
		rate = 1
	}
	return func(args ...interface{}) int {
		start := time.Now()
		status := inner(args...)
		callTime := time.Since(start)

		statLine := fmt.Sprintf("%s.status.%d", name, status)
		_ = s.stats.Inc(statLine, 1, rate)
		s.cloud.Inc(statLine, 1)

		timeLine := fmt.Sprintf("%s.timing", name)
		_ = s.stats.TimingDuration(timeLine, callTime, rate)
		s.cloud.Timing(timeLine, callTime)

		statTimeLine := fmt.Sprintf("%s.%d", timeLine, status)
		_ = s.stats.TimingDuration(statTimeLine, callTime, rate)
		s.cloud.Timing(statTimeLine, callTime)

		return status
	}
}
