package params

import (
	"context"
	"encoding/json"
	"io"
	"io/ioutil"
	"net/http"

	"github.com/jixwanwang/apiutils"

	"goji.io/pattern"
)

var (
	ErrMissingBody = apiutils.NewErrorResponse(http.StatusBadRequest, "No body found")
)

var bodyKey int

type Params struct {
	ctx context.Context
}

func New(ctx context.Context) *Params {
	return &Params{ctx}
}

func Set(ctx context.Context, name string, value interface{}) context.Context {
	return context.WithValue(ctx, pattern.Variable(name), value)
}

func SetBody(ctx context.Context, body io.ReadCloser) context.Context {
	return context.WithValue(ctx, &bodyKey, body)
}

func (p *Params) Get(name string) string {
	str := p.GetOptional(name)
	if str == nil {
		return ""
	}
	return *str
}

func (p *Params) GetObject(name string) interface{} {
	return p.ctx.Value(pattern.Variable(name))
}

func (p *Params) GetBody() io.ReadCloser {
	if body, ok := p.ctx.Value(&bodyKey).(io.ReadCloser); ok {
		return body
	}
	return nil
}

func (p *Params) ParseJSONBody(out interface{}) error {
	body := p.GetBody()
	if body == nil {
		return ErrMissingBody
	}

	buf, err := ioutil.ReadAll(body)
	if err != nil {
		return err
	}

	if len(buf) == 0 {
		return ErrMissingBody
	}

	return json.Unmarshal(buf, out)
}

func (p *Params) GetOptional(name string) *string {
	value := p.GetObject(name)
	switch value.(type) {
	case string:
		str := value.(string)
		return &str
	default:
		return nil
	}
}
