package uuid

import (
	"crypto/rand"
	"fmt"
	"io"
)

// Source is responsible for returning version 4 UUIDs on demand
type Source interface {
	Next() (string, error)
}

type uuidGenerator struct {
	randSource io.Reader
}

// NewSource creates a Source wrapped around the standard rand.Reader
// which is suitable for non-cryptographic uses such as optimistic lock input
func NewSource() Source {
	return &uuidGenerator{rand.Reader}
}

func NewCustomSource(reader io.Reader) Source {
	return &uuidGenerator{reader}
}

// ok to use local random here for optimistic lock input; doesn't require crypto strength
func (u *uuidGenerator) Next() (string, error) {
	raw := make([]byte, 16)
	_, err := u.randSource.Read(raw)
	if err != nil {
		return "", err
	}
	raw[6] = (raw[6] | 0x80) & 0xBF
	raw[8] = (raw[8] | 0x80) & 0xBF
	return fmt.Sprintf("%X-%X-%X-%X-%X", raw[0:4], raw[4:6], raw[6:8], raw[8:10], raw[10:]), nil
}
