package manager

import (
	"context"
	"log"
	"net/http"
	"strconv"
	"time"

	"code.justin.tv/systems/sandstorm/closer"
	"code.justin.tv/systems/sandstorm/internal/envelope/envelopeiface"
	"code.justin.tv/systems/sandstorm/internal/stat"
	"code.justin.tv/systems/sandstorm/internal/stat/statiface"
	"code.justin.tv/systems/sandstorm/inventory/heartbeat"
	"code.justin.tv/systems/sandstorm/logging"
	"code.justin.tv/systems/sandstorm/queue"
	"code.justin.tv/systems/sandstorm/resource"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/service/cloudwatch"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"
	"github.com/aws/aws-sdk-go/service/kms/kmsiface"
	"github.com/aws/aws-sdk-go/service/sns"
	"github.com/aws/aws-sdk-go/service/sqs"
	uuid "github.com/satori/go.uuid"
)

// API describes the manager interface
type API interface {
	AuditTableName() string
	CheckKMS() error
	CheckTable() error
	Copy(source, destination string) error
	CreateTable() error
	Decrypt(secret *Secret) error
	Delete(secretName string) error
	Exist(secretName string) (bool, error)
	CrossEnvironmentSecretsSet(secretNames []string) (crossEnvSecrets map[string]struct{}, err error)
	FlushHeartbeat(ctx context.Context)
	Get(secretName string) (*Secret, error)
	GetEncrypted(secretName string) (*Secret, error)
	GetVersion(secretName string, version int64) (*Secret, error)
	GetVersionsEncrypted(secretName string, limit int64, offsetKey int64) (*VersionsPage, error)
	List() ([]*Secret, error)
	ListenForUpdates() error
	ListNamespace(namespace string) ([]*Secret, error)
	ListNamespaces() ([]string, error)
	NamespaceTableName() string
	Patch(patchInput *PatchInput) error
	Post(secret *Secret) error
	Put(secret *Secret) error
	Revert(name string, version int64) error
	RegisterSecretUpdateCallback(secretName string, callback func(input SecretChangeset))
	Seal(secret *Secret) error
	StopListeningForUpdates() error
	TableName() string
	CleanUp() error

	DeleteSecret(input *DeleteSecretInput) (output *DeleteSecretOutput, err error)
	RevertSecret(input *RevertSecretInput) (output *RevertSecretOutput, err error)
	CopySecret(input *CopySecretInput) (output *CopySecretOutput, err error)
}

// InputValidationError represents error if input was invalid
type InputValidationError struct {
	err string
}

func (e *InputValidationError) Error() string {
	return e.err
}

const version = "1.0"

const (
	dynamoDBKeyTombstone       = "tombstone"
	dynamoDBKeyActionUser      = "action_user"
	dynamoDBKeyServiceName     = "service_name"
	dynamoDBKeyPreviousVersion = "previous_version"
)

const (
	attempted = "attempted"
	success   = "success"
	failure   = "failure"
	cached    = "cached"
	notFound  = "not_found"
)

const awsRegionUsWest2 = "us-west-2"

// Manager is the struct that a caller interacts with for managing
// secrets.
type Manager struct {
	DynamoDB                dynamodbiface.DynamoDBAPI
	Config                  Config
	Logger                  logging.Logger
	statter                 statiface.API
	inventoryClient         heartbeat.API
	queue                   queue.Queuer
	cache                   *cache
	stopListeningForUpdates chan struct{}
	secretCallbacks         map[string][]func(SecretChangeset)

	clientLoader *clientLoader
	closer       *closer.Closer
	enveloper    envelopeiface.Enveloper
}

// Config is a struct for configuring the manager
// XXX need to be able to provide an aws.Config for STS credentials,
// region etc
type Config struct {
	// AWS config structure. If left nil, will be replaced by a
	// default config with region us-west-2.
	AWSConfig *aws.Config
	// KMS KeyId - this is either a key alias prefixed by 'alias/',
	// a key ARN or a key UUID
	KeyID string
	// DynamoDB Table name
	TableName string
	// Provisioned Read and Write throughput when creating the
	// table, not used in normal operations. If left to 0 these
	// will be replaced by 1 by default.
	ProvisionedThroughputRead  int64
	ProvisionedThroughputWrite int64
	Logger                     logging.Logger

	Host               string
	InventoryRoleARN   string
	CloudWatchRoleARN  string
	InventoryStatusURL string
	InventoryInterval  time.Duration
	ServiceName        string

	Queue      queue.Config
	InstanceID string

	//User performing the action
	ActionUser string

	// Environment to configure manager for
	Environment string

	// ProxyURL [Optional] proxy to use as a backUp http transport
	ProxyURL string

	// Pop [Optional] name of the pop manager is running in.
	Pop string

	// Region
	primaryRegion string

	kmsKeys []resource.KMSKey
}

func (cfg Config) awsConfigurerForRegion(region string) resource.AWSConfigurer {
	awsConfigurer := resource.AWSConfigurer{
		Region: region,
	}

	if cfg.AWSConfig != nil {
		awsConfigurer.HTTPClient = cfg.AWSConfig.HTTPClient
	}

	return awsConfigurer
}

func (cfg Config) awsConfigurer() resource.AWSConfigurer {
	return cfg.awsConfigurerForRegion(cfg.primaryRegion)
}

func (cfg Config) awsCreds() *credentials.Credentials {
	if cfg.AWSConfig == nil {
		return nil
	}
	return cfg.AWSConfig.Credentials
}

const (
	unknownServiceName     = "unknown-service"
	defaultQueueNamePrefix = "sandstorm"
)

// DefaultConfig returns a config struct with some sane defaults that
// is merged with the provided config in New
func DefaultConfig() Config {
	defEnv := "production"

	// gets a predefined config values for production env.
	// This call with arg "" or "production" should not return error.
	res, err := resource.GetConfigForEnvironment(defEnv)
	if err != nil {
		log.Fatal(err.Error())
	}

	return Config{
		Environment:                defEnv,
		ProvisionedThroughputRead:  1,
		ProvisionedThroughputWrite: 1,
		AWSConfig:                  resource.AWSConfig(nil),
		Logger:                     &logging.NoopLogger{},
		InventoryRoleARN:           res.InventoryRoleARN,
		CloudWatchRoleARN:          res.CloudwatchRoleArn,
		KeyID:                      res.KMSKeyID,
		TableName:                  res.TableName,
		Queue: queue.Config{
			Environment:     defEnv,
			QueueNamePrefix: defaultQueueNamePrefix,
		},
		primaryRegion: res.AwsRegion,
		kmsKeys:       res.KMSKeys,
	}
}

func getDefaultConfigByEnvironment(env string) (cfg Config) {

	cfg = DefaultConfig()
	if env == "" || env == "production" {
		return
	}

	// fetchs config values for the specified env stored in s3.
	res, err := resource.GetConfigForEnvironment(env)
	if err != nil {
		log.Fatal(err.Error())
	}

	cfg.InventoryRoleARN = res.InventoryRoleARN
	cfg.CloudWatchRoleARN = res.CloudwatchRoleArn
	cfg.KeyID = res.KMSKeyID
	cfg.TableName = res.TableName
	cfg.Logger = &logging.NoopLogger{}
	cfg.Queue = queue.Config{
		Environment:     env,
		QueueNamePrefix: defaultQueueNamePrefix,
	}
	cfg.kmsKeys = res.KMSKeys
	return
}

func httpClientWithProxy(config *Config, statter ProxyStatter) (httpClient *http.Client) {

	rt := &wrappedRoundTripper{
		Logger:   config.Logger,
		Pop:      config.Pop,
		ProxyURL: config.ProxyURL,
		Statter:  statter,
	}

	httpClient = &http.Client{Transport: rt}
	return
}

func mergeConfig(provided *Config, defConfig Config) {
	if provided.ProvisionedThroughputRead == 0 {
		provided.ProvisionedThroughputRead = defConfig.ProvisionedThroughputRead
	}
	if provided.ProvisionedThroughputWrite == 0 {
		provided.ProvisionedThroughputWrite = defConfig.ProvisionedThroughputWrite
	}
	if provided.Logger == nil {
		provided.Logger = defConfig.Logger
	}
	if provided.InventoryRoleARN == "" {
		provided.InventoryRoleARN = defConfig.InventoryRoleARN
	}
	if provided.CloudWatchRoleARN == "" {
		provided.CloudWatchRoleARN = defConfig.CloudWatchRoleARN
	}
	if provided.InstanceID == "" {
		provided.InstanceID = uuid.NewV4().String()
	}
	if provided.KeyID == "" {
		provided.KeyID = defConfig.KeyID
	}
	if provided.TableName == "" {
		provided.TableName = defConfig.TableName
	}
	if provided.AWSConfig == nil {
		provided.AWSConfig = defConfig.AWSConfig
	}
	if provided.ActionUser == "" {
		provided.ActionUser = provided.awsConfigurer().GetAWSIdentity(provided.awsCreds())
	}
	if provided.Queue == (queue.Config{}) {
		provided.Queue = defConfig.Queue
	}
	// Set the private configs.
	provided.kmsKeys = defConfig.kmsKeys
	provided.primaryRegion = defConfig.primaryRegion
}

// New creates a Manager from config. Merges configuration with
// DefaultConfig()
func New(config Config) *Manager {
	return NewWithProxyStatter(config, nil)
}

// NewWithProxyStatter creates a managee with statter to report
// proxy usage of manager if proxy is configured
func NewWithProxyStatter(config Config, statter ProxyStatter) *Manager {

	defConfig := getDefaultConfigByEnvironment(config.Environment)
	mergeConfig(&config, defConfig)

	//Setup http client  proxy
	if config.ProxyURL != "" {
		httpClient := httpClientWithProxy(&config, statter)
		config.AWSConfig.WithHTTPClient(httpClient)
	}

	sess := config.awsConfigurer().STSSession(config.awsCreds())

	clientLoader := &clientLoader{
		Config: config,
	}

	closer := closer.New()

	cwStatter := &stat.Client{
		Namespace: "Sandstorm",
		Logger:    config.Logger,
		CloudWatch: cloudwatch.New(config.awsConfigurer().STSSession(
			config.awsCreds(), config.CloudWatchRoleARN)),
		Closer: closer,
	}

	mgr := &Manager{
		Config:  config,
		Logger:  config.Logger,
		statter: cwStatter,
		cache:   &cache{},
		queue:   queue.New(sqs.New(sess), sns.New(sess), config.Queue, config.Logger),
		stopListeningForUpdates: make(chan struct{}),
		secretCallbacks:         make(map[string][]func(input SecretChangeset)),
		clientLoader:            clientLoader,
		closer:                  closer,
		enveloper: &Envelope{
			clientLoader:  clientLoader,
			primaryRegion: config.primaryRegion,
			kmsKeys:       config.kmsKeys,
			Logger:        config.Logger,
			statter:       cwStatter,
		},
	}

	mgr.inventoryClient = heartbeat.New(
		config.awsConfigurer().AWSCredentials(config.awsCreds(), config.InventoryRoleARN),
		&heartbeat.Config{
			Interval:       config.InventoryInterval,
			Service:        config.ServiceName,
			Host:           config.Host,
			InstanceID:     config.InstanceID,
			ManagerVersion: version,
			Environment:    config.Environment,
		},
		config.Logger,
	)
	go mgr.inventoryClient.Start()

	return mgr
}

func (mgr *Manager) newCache() {
	mgr.cache = &cache{
		secretMap: make(map[string]*Secret),
	}
}

// ListenForUpdates manager listens on the sandstorm queue for secret updates.
// It also configures the manager to cache secret plaintext in memory. When an
// update event is received for a secret, that secret is expunged from the in
// memory cache, so the next Get call will fetch that secret from DynamoDB.
func (mgr *Manager) ListenForUpdates() (err error) {
	mgr.newCache()
	err = mgr.queue.Setup()
	if err != nil {
		return
	}

	go func() {
		for {
			select {
			case <-mgr.stopListeningForUpdates:
				return
			default:
				err := mgr.listenForSecretUpdates()
				if err != nil {
					mgr.Logger.Errorf("Failed to listen for secret update. err: %s", err.Error())
				}
			}
		}
	}()
	return
}

func (mgr *Manager) listenForSecretUpdates() (err error) {
	errChan := make(chan error)
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	go func() {
		queueSecret, err := mgr.queue.PollForSecret(ctx)
		if err != nil {
			errChan <- err
			return
		}
		if queueSecret != nil {
			secretName := queueSecret.Name.S
			mgr.cache.Delete(secretName)
			// This should always be int, if for some reason conversion fails,
			// send the error over to chan and continue with updatedAt = 0
			var updatedAt int64
			if queueSecret.UpdatedAt.S != "" {
				updatedAt, err = strconv.ParseInt(queueSecret.UpdatedAt.S, 10, 64)
				if err != nil {
					errChan <- err
				}
			}
			mgr.triggerSecretUpdateCallbacks(secretName, updatedAt)
		}
		errChan <- nil
		return
	}()

	select {
	case <-mgr.stopListeningForUpdates:
		err = <-errChan
	case err = <-errChan:
	}
	return
}

// StopListeningForUpdates manager stops listening to secret update queue and
// deletes the queue
func (mgr *Manager) StopListeningForUpdates() (err error) {
	mgr.cache = &cache{}
	close(mgr.stopListeningForUpdates)
	return mgr.queue.Delete()
}

// CleanUp cleans exits any go routines spawned
func (mgr *Manager) CleanUp() (err error) {
	if err := mgr.closer.Close(); err != nil {
		mgr.Logger.Errorf("error closing manager: %s", err.Error())
	}

	inventoryStopErr := mgr.inventoryClient.Stop()
	if inventoryStopErr != nil {
		mgr.Logger.Debugf("error stopping inventory client: %s", inventoryStopErr.Error())
	}
	return
}

// FlushHeartbeat sends a signal to immediately report heartbeat
func (mgr *Manager) FlushHeartbeat(ctx context.Context) {
	mgr.inventoryClient.FlushHeartbeat(ctx)
}

func (mgr *Manager) dynamoDB() dynamodbiface.DynamoDBAPI {
	return mgr.clientLoader.DynamoDB(mgr.Config.primaryRegion)
}

func (mgr *Manager) kms() kmsiface.KMSAPI {
	return mgr.clientLoader.KMS(mgr.Config.primaryRegion)
}
