import * as fs from "fs";
import * as path from "path";
import { inspect } from "util";
import { compile as compileTemplate, SafeString } from "handlebars";
import { StaticTokenMap, DynamicThemeMap } from "twitch-core-ui-tokens";


interface ThemeNamespace {
    namespace: string,
    values: Array<string>
}

interface ThemeHierarchy {
    [key: string]: ThemeNamespace | Array<ThemeNamespace>,
}

interface ThemeConfig {
    [key: string]: ThemeValue
}

interface ThemeValue {
    [key: string]: string
}

DynamicThemeMap.dark["color-border"]

const theme: ThemeHierarchy = {
    backgroundColor: {
        namespace: "color-background",
        values: ["body", "base", "alt", "alt-2", "accent", "accent-alt", "accent-alt-2"]
    },

    borderColor: {
        namespace: "color-border",
        values: ["alert", "base", "button", "error", "input", "quote", "selectable"]
    },
    fill: {
        namespace: "color-fill",
        values: ["base", "alt", "alt-2", "brand", "offline", "live", "info", "warn", "alert", "success"]
    },
    textColor: {
        namespace: "color-text",
        values: ["base", "alt", "alt2", "link"]
    },
}

type ThemeBuilder = (currentValue: ThemeValue, nextValue: string) => ThemeValue;
function buildConfigForNamespace(namespace: string): ThemeBuilder {
    return function(acc: {}, key: string) {
        acc[key] = `var(--${namespace}-${key})`;
        return acc;
    }
}

function buildConfigForLevel(value: ThemeValue, ns: ThemeNamespace): ThemeValue {
    return ns.values.reduce(buildConfigForNamespace(ns.namespace), value);
}

function buildConfigFromTheme(hierarchy: ThemeHierarchy): ThemeConfig {
    return Object.keys(hierarchy).reduce((config: ThemeConfig, propName: string) => {
        const props = hierarchy[propName];
        let value = config[propName] || {};

        if (props instanceof Array) {
            props.reduce(buildConfigForLevel, value);
        } else {
            value = buildConfigForLevel(value, props);
        }

        config[propName] = value;
        return config;
    }, {});
}

interface PropertySet {
    [key: string]: string
}

interface PropertySetOptions {
    without: Array<string>,
}

function extractPropertySet(baseName: string, options?: PropertySetOptions): PropertySet {
    const baseKey = `${baseName}-`;
    return Object.keys(StaticTokenMap).filter((k) => k.startsWith(baseKey)).reduce((acc, k) => {
        const key = k.replace(baseKey, "");
        if (options?.without.includes(key)) {
            return acc;
        }

        const value = StaticTokenMap[k];
        acc[key] = value;
        return acc;
    }, {});
}

const baseThemeConfig = buildConfigFromTheme(theme);
const themeConfig = {
    ...baseThemeConfig,
    backgroundColor: {
        ...baseThemeConfig.backgroundColor,
    },
    borderWidth: extractPropertySet('border-width', {
        without: ["input-overlay-unfocused", "spinner", "tag"]
    }),
    borderRadius: extractPropertySet('border-radius'),
    colors: {
        transparent: StaticTokenMap["color-transparent"],
        black: StaticTokenMap["color-black"],
        white: StaticTokenMap["color-white"],

        accent: StaticTokenMap["color-accent"],
        error: StaticTokenMap["color-error"],
        info: StaticTokenMap["color-info"],
        success: StaticTokenMap["color-success"],
        warn: StaticTokenMap["color-warn"],

        blue: StaticTokenMap["color-blue"],
        green: {
            normal: StaticTokenMap["color-green"],
            darker: StaticTokenMap["color-green-darker"],
        },
        magenta: StaticTokenMap["color-magenta"],
        orange: StaticTokenMap["color-orange"],

        'hinted-gray': extractPropertySet('color-hinted-grey'),
        'creator-accent': extractPropertySet('color-brand-accent'),
        'creator-muted': extractPropertySet('color-brand-muted'),
        purple: extractPropertySet('color-twitch-purple'),

        'opac-b': extractPropertySet('color-opac-b'),
        'opac-w': extractPropertySet('color-opac-w'),
        'opac-p': extractPropertySet('color-opac-p'),
    },
    fontFamily: {
        sans: StaticTokenMap["font-base"],
        header: StaticTokenMap["font-display"],
        mono: StaticTokenMap["font-mono"],
    },
    fontSize: extractPropertySet('font-size'),
    fontWeight: extractPropertySet('font-weight'),
    screens: extractPropertySet('break'),
    spacing: extractPropertySet('space', {
        without: ["auto"],
    }),
    zIndex: extractPropertySet('z-index'),
}

fs.writeFileSync(
    path.join(
        process.cwd(),
        "tailwind.config.js",
    ),
    compileTemplate(
        fs.readFileSync(path.join(__dirname, "tailwind.config.template.hbs"), "utf8"),
    )({
        globals: StaticTokenMap,
        themes: DynamicThemeMap,
        themeObjectLiteral: new SafeString(inspect(themeConfig, {
            depth: Number.POSITIVE_INFINITY,
            sorted: false,
        })),
    }),
);

fs.mkdirSync("src/styles/generated");

fs.writeFileSync(
    path.join(
        process.cwd(),
        "src/styles/generated/core-ui-aliases.scss",
    ),
    compileTemplate(
        fs.readFileSync(path.join(__dirname, "core-ui-aliases.scss.hbs"), "utf8"),
    )({
        spacing: themeConfig.spacing,
        sides: ["l", "t", "r", "b"]
    })
)