/**
 * Simple wrapper script to avoid thinking about how to run our scripts
 * Finds a script with the given name regardless of its extension and runs
 * it through ts-node. Script may be a ts file, js file, or folder with an
 * index.js or index.ts file inside.
 */
import * as spawn from "cross-spawn";
import * as fs from "fs";
import { sync as globSync } from "glob";
import * as path from "path";

const program = process.argv[2];
let didRun = false;

if (program) {
  const projectPath = path.dirname(__dirname);
  const targets = [
    ...globSync(program + "+(.ts|.js)", { cwd: __dirname }),
    ...globSync(program + "/index+(.ts|.js)", { cwd: __dirname }),
  ];

  if (targets.length > 1) {
    let err = "Found multiple possible targets for given script name:\n";
    for (const target of targets) {
      err += `  scripts/${target}\n`;
    }
    err += "Overlapping script targets should be removed or renamed.";
    process.exit(1);
  }

  try {
    const target = path.join(__dirname, targets[0]);
    // double check access to target just to be sure
    fs.accessSync(target);
    const child = spawn(
      path.join(projectPath, "node_modules", ".bin", "ts-node"),
      ["--project", "scripts/tsconfig.json", target, ...process.argv.slice(3)],
      { stdio: "inherit" }
    );
    child.on("exit", (code) => {
      process.exit(code || 0);
    });
    didRun = true;
  } catch (e) {
    // Not interested in catching these, we will fall through
    // to the usage output below if script does not execute.
  }
}

if (!didRun) {
  console.error("Script not found. Available scripts:");

  const topLevel = globSync("+(*.js|*.ts)", { cwd: __dirname }).map(
    (file) => file.split(".")[0]
  );
  const directories = globSync("*/index.+(js|ts)", { cwd: __dirname }).map(
    (folder) => folder.split(path.sep)[0]
  );

  for (const script of topLevel.concat(directories).sort()) {
    if (script === "run-script") {
      continue;
    }

    console.log("  " + script);
  }
}
