package auth

// ExtensionRole is a shorthand for extension state transition permissions
type ExtensionRole int

const (
	// NoRole means this user has no elevated privileges
	NoRole ExtensionRole = 0
	// Developer is the 3rd party that owns extension(s) - id list should be consulted
	Developer ExtensionRole = 1 << 0
	// Reviewer is a person who can review an extension
	Reviewer ExtensionRole = 1 << 1
	// Moderator is a person who can reject an extension, killing it
	Moderator ExtensionRole = 1 << 2
	// Cleric is a person who can revive deleted extensions
	Cleric ExtensionRole = 1 << 3
	// Creator is a person who can make new extensions
	Creator ExtensionRole = 1 << 4
	// Administrator is a person who can edit whitelists
	Administrator ExtensionRole = 1 << 5
	// Categorizer is a person who can view and edit categories and games
	Categorizer ExtensionRole = 1 << 6
	// Seer is somebody who can view any extension before release
	Seer ExtensionRole = 1 << 7
	// Authorizer is a caller who can view whitelists
	Authorizer ExtensionRole = 1 << 8
	// Installer is a caller who can install anything they see
	Installer ExtensionRole = 1 << 9
	// Monetizer is a user who can monetize extensions
	Monetizer ExtensionRole = 1 << 10
)

// IsMember allows testing if Credentials are equivalent any active bit in the specified role.
func (role ExtensionRole) IsMember(extensionID string, creds Credentials) bool {
	isMember := false
	if (role & Developer) == Developer {
		isMember = isMember || creds.CanEditVersion(extensionID)
	}
	if (role & Reviewer) == Reviewer {
		isMember = isMember || creds.CanReviewExtensions()
	}
	if (role & Moderator) == Moderator {
		isMember = isMember || creds.CanModerateExtensions()
	}
	if (role & Cleric) == Cleric {
		isMember = isMember || creds.CanReviveExtensions()
	}
	if (role & Creator) == Creator {
		isMember = isMember || creds.CanCreateExtensions()
	}
	if (role & Seer) == Seer {
		isMember = isMember || creds.CanViewAllExtensions()
	}
	if (role & Installer) == Installer {
		isMember = isMember || creds.CanInstallAllExtensions()
	}
	if (role & Monetizer) == Monetizer {
		isMember = isMember || creds.CanMonetizeExtensions()
	}
	return isMember
}
