package logmiddleware

import (
	"code.justin.tv/extensions/discovery/golibs/logger"
	"context"
	"encoding/json"

	"github.com/twitchtv/twirp"
)

// Server middleware for logging errors that occur when handling requests.
type Server struct {
	// Encoder is used to encode log messages
	Encoder *json.Encoder
	// These error codes will not be logged.
	SuppressedErrorCodes []twirp.ErrorCode
}

type ErrorLogMessage struct {
	Msg             string             `json:"msg"`
	Stack           *logger.TraceChain `json:"Stack"`
	Method          string             `json:"Method"`
	Service         string             `json:"Service"`
	HTTPStatusCode  string             `json:"HTTPStatusCode"`
	TwirpStatusCode string             `json:"TwirpStatusCode"`
}

// ServerHooks returns Twirp hooks for logging errors.
func (s *Server) ServerHooks() *twirp.ServerHooks {
	return &twirp.ServerHooks{
		Error: func(ctx context.Context, err twirp.Error) context.Context {
			s.Log(ctx, err)
			return ctx
		},
	}
}

func (s *Server) Log(ctx context.Context, err twirp.Error) {
	if s.Encoder == nil || err == nil {
		return
	}

	errCode := err.Code()
	for _, suppressed := range s.SuppressedErrorCodes {
		if errCode == suppressed {
			return
		}
	}

	methodName, _ := twirp.MethodName(ctx)
	serviceName, _ := twirp.ServiceName(ctx)
	statusCode, _ := twirp.StatusCode(ctx)

	logMessage := ErrorLogMessage{
		Msg:             "Error while handling request: " + err.Error(),
		Method:          methodName,
		HTTPStatusCode:  statusCode,
		Service:         serviceName,
		TwirpStatusCode: string(errCode),
		Stack:           logger.GetStackChain(err),
	}

	_ = s.Encoder.Encode(logMessage)
}
