package main

import (
	"context"
	"testing"

	"code.justin.tv/extensions/discovery/auth"
	"code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/discovery/data"
	"code.justin.tv/extensions/discovery/data/model/memory"
	"code.justin.tv/extensions/discovery/golibs/uuid"
	"code.justin.tv/extensions/discovery/manager"
	"code.justin.tv/extensions/discovery/twirputils"

	"github.com/stretchr/testify/assert"
)

func TestAddCategory(t *testing.T) {
	store := memory.New(uuid.NewSource())
	discoveryService := fakeServer(t, store)

	t.Run("Must have a body", func(t *testing.T) {
		ctx := context.Background()

		var req *discovery.AddCategoryRequest

		out, err := discoveryService.AddCategory(ctx, req)
		assert.Nil(t, out)
		assert.Equal(t, data.ErrMissingParameter, err)
	})

	t.Run("Must have a category", func(t *testing.T) {
		ctx := context.Background()

		req := &discovery.AddCategoryRequest{}

		out, err := discoveryService.AddCategory(ctx, req)
		assert.Nil(t, out)
		assert.Equal(t, data.ErrMissingParameter, err)
	})

	t.Run("Must have a store", func(t *testing.T) {
		ctx := context.Background()

		req := &discovery.AddCategoryRequest{
			Category: &discovery.CategoryUpsert{
				Type:    twirputils.WrapString("developer"),
				SortKey: twirputils.WrapString("manual"),
			},
		}

		out, err := discoveryService.AddCategory(ctx, req)
		assert.Nil(t, out)
		assert.Equal(t, data.ErrUnavailable, err)
	})

	t.Run("Must be allowed to edit categories", func(t *testing.T) {
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))

		req := &discovery.AddCategoryRequest{
			Category: &discovery.CategoryUpsert{
				Type:    twirputils.WrapString("developer"),
				SortKey: twirputils.WrapString("manual"),
			},
		}

		out, err := discoveryService.AddCategory(ctx, req)
		assert.Nil(t, out)
		assert.Equal(t, data.ErrUnauthorized, err)
	})

	t.Run("Category type must make sense", func(t *testing.T) {
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})
		req := &discovery.AddCategoryRequest{
			Category: &discovery.CategoryUpsert{
				Type:    twirputils.WrapString("nonsense"),
				SortKey: twirputils.WrapString("manual"),
			},
		}

		out, err := discoveryService.AddCategory(ctx, req)
		assert.Nil(t, out)
		assert.Equal(t, data.ErrUnknownCategoryType, err)
	})

	t.Run("Category sort key must make sense", func(t *testing.T) {
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})
		req := &discovery.AddCategoryRequest{
			Category: &discovery.CategoryUpsert{
				Type:    twirputils.WrapString("developer"),
				SortKey: twirputils.WrapString("nonsense"),
			},
		}

		out, err := discoveryService.AddCategory(ctx, req)
		assert.Nil(t, out)
		assert.Equal(t, data.ErrUnknownCategorySortKey, err)
	})

	t.Run("Can create a category successfully", func(t *testing.T) {
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})
		req := &discovery.AddCategoryRequest{
			Category: &discovery.CategoryUpsert{
				Type:    twirputils.WrapString("developer"),
				SortKey: twirputils.WrapString("manual"),
			},
		}

		out, err := discoveryService.AddCategory(ctx, req)
		assert.NotNil(t, out)
		assert.Nil(t, err)
	})
}
