package main

import (
	"context"
	"testing"

	"code.justin.tv/extensions/discovery/auth"
	"code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/discovery/data"
	"code.justin.tv/extensions/discovery/data/model"
	"code.justin.tv/extensions/discovery/data/model/memory"
	"code.justin.tv/extensions/discovery/golibs/uuid"
	"code.justin.tv/extensions/discovery/manager"
	"code.justin.tv/extensions/discovery/twirputils"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

const (
	testExtensionID = "testExtensionID"
	testVersionID   = "testVersionID"
)

func TestAddDevCategoryToExtension(t *testing.T) {
	t.Run("must have request", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)

		ctx := context.Background()
		out, err := discoveryService.AddCategoryToExtension(ctx, nil)
		assert.Nil(t, out)
		assert.Equal(t, data.ErrMissingParameter, err)
	})

	t.Run("must have categoryID set", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)

		ctx := manager.StoreDiscoveryManager(context.Background(), buildTestDiscoveryManager(store))
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		resp, err := discoveryService.AddCategoryToExtension(ctx, &discovery.AddCategoryToExtensionRequest{
			ExtensionID: testExtensionID,
			VersionID:   testVersionID,
		})
		assert.Nil(t, resp)
		assert.Equal(t, data.ErrMissingParameter, err)
	})

	t.Run("manager must be set on context", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)

		ctx := context.Background()
		out, err := discoveryService.AddCategoryToExtension(ctx, &discovery.AddCategoryToExtensionRequest{
			CategoryID:  "catid",
			ExtensionID: testExtensionID,
			VersionID:   testVersionID,
		})
		assert.Nil(t, out)
		assert.Equal(t, data.ErrUnavailable, err)
	})

	t.Run("category must exist", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)

		ctx := manager.StoreDiscoveryManager(context.Background(), buildTestDiscoveryManager(store))
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		resp, err := discoveryService.AddCategoryToExtension(ctx, &discovery.AddCategoryToExtensionRequest{
			CategoryID:  "nocat",
			ExtensionID: testExtensionID,
			VersionID:   testVersionID,
		})
		assert.Nil(t, resp)
		assert.Equal(t, data.ErrNotFound, err)
	})

	t.Run("with category must have ext/version set", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)

		ctx := manager.StoreDiscoveryManager(context.Background(), buildTestDiscoveryManager(store))
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		cat, err := store.CreateCategory(&discovery.AddCategoryRequest{
			Category: &discovery.CategoryUpsert{
				Type:    twirputils.WrapString(model.CategoryTypeDeveloper),
				SortKey: twirputils.WrapString(model.CategorySortkeyManual),
				Visible: twirputils.WrapBool(true),
			},
		})
		require.NoError(t, err)

		resp, err := discoveryService.AddCategoryToExtension(ctx, &discovery.AddCategoryToExtensionRequest{
			CategoryID: cat.ID,
			VersionID:  testVersionID,
		})
		assert.Nil(t, resp)
		assert.Equal(t, data.ErrMissingParameter, err)

		resp, err = discoveryService.AddCategoryToExtension(ctx, &discovery.AddCategoryToExtensionRequest{
			CategoryID:  cat.ID,
			ExtensionID: testExtensionID,
		})
		assert.Nil(t, resp)
		assert.Equal(t, data.ErrMissingParameter, err)
	})

	t.Run("must be authorized", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)

		ctx := manager.StoreDiscoveryManager(context.Background(), buildTestDiscoveryManager(store))
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.NoRole})

		cat, err := store.CreateCategory(&discovery.AddCategoryRequest{
			Category: &discovery.CategoryUpsert{
				Type:    twirputils.WrapString(model.CategoryTypeDeveloper),
				SortKey: twirputils.WrapString(model.CategorySortkeyManual),
				Visible: twirputils.WrapBool(true),
			},
		})
		require.NoError(t, err)

		resp, err := discoveryService.AddCategoryToExtension(ctx, &discovery.AddCategoryToExtensionRequest{
			CategoryID:  cat.ID,
			ExtensionID: testExtensionID,
			VersionID:   testVersionID,
		})
		assert.Nil(t, resp)
		assert.Equal(t, data.ErrUnauthorized, err)
	})

	t.Run("disco data must exist", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)

		ctx := manager.StoreDiscoveryManager(context.Background(), buildTestDiscoveryManager(store))
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Seer | auth.Categorizer})

		cat, err := store.CreateCategory(&discovery.AddCategoryRequest{
			Category: &discovery.CategoryUpsert{
				Type:    twirputils.WrapString(model.CategoryTypeDeveloper),
				SortKey: twirputils.WrapString(model.CategorySortkeyManual),
				Visible: twirputils.WrapBool(true),
			},
		})
		require.NoError(t, err)

		resp, err := discoveryService.AddCategoryToExtension(ctx, &discovery.AddCategoryToExtensionRequest{
			CategoryID:  cat.ID,
			ExtensionID: testExtensionID,
			VersionID:   testVersionID,
		})
		assert.Nil(t, resp)
		assert.Equal(t, data.ErrNotFound, err)
	})

	t.Run("success", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)

		ctx := manager.StoreDiscoveryManager(context.Background(), buildTestDiscoveryManager(store))
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Seer | auth.Categorizer})

		cat, err := store.CreateCategory(&discovery.AddCategoryRequest{
			Category: &discovery.CategoryUpsert{
				Type:    twirputils.WrapString(model.CategoryTypeDeveloper),
				SortKey: twirputils.WrapString(model.CategorySortkeyManual),
				Visible: twirputils.WrapBool(true),
			},
		})
		require.NoError(t, err)

		discoData, err := store.PutDiscoveryData(&discovery.PutExtensionVersionDiscoveryDataRequest{
			UpdateData: &discovery.ExtensionVersionDiscoveryDataUpdate{
				ExtensionID: testExtensionID,
				Version:     testVersionID,
			},
		})
		require.NoError(t, err)

		resp, err := discoveryService.AddCategoryToExtension(ctx, &discovery.AddCategoryToExtensionRequest{
			CategoryID:  cat.ID,
			ExtensionID: discoData.ExtensionID,
			VersionID:   discoData.Version,
		})

		require.Nil(t, err)
		assert.Equal(t, cat.ID, resp.GetExtensionCategoryMembership().GetCategoryID())
		assert.Equal(t, testExtensionID, resp.GetExtensionCategoryMembership().GetExtensionID())
		assert.Nil(t, resp.GetExtensionCategoryMembership().GetOrder())
		assert.Nil(t, resp.GetExtensionCategoryMembership().GetStartTime())
	})
}
