package main

import (
	"context"
	"testing"

	"code.justin.tv/extensions/discovery/auth"
	discovery "code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/discovery/data"
	"code.justin.tv/extensions/discovery/data/model"
	"code.justin.tv/extensions/discovery/data/model/memory"
	"code.justin.tv/extensions/discovery/golibs/uuid"
	"code.justin.tv/extensions/discovery/manager"
	"code.justin.tv/extensions/discovery/twirputils"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

var defaultOrder = float32(0)

func TestAddExtensionToCategory(t *testing.T) {
	t.Run("must have request", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)

		ctx := context.Background()
		out, err := discoveryService.AddExtensionToCategory(ctx, nil)
		assert.Nil(t, out)
		assert.Equal(t, data.ErrMissingParameter, err)
	})

	t.Run("manager must be set on context", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)

		ctx := auth.Store(context.Background(), &auth.FakeCredentials{Role: auth.Categorizer})

		req := makeAddExtensionToCategoryRequest("testExtensionID", "testCategoryID", "testVersionID", &defaultOrder)
		out, err := discoveryService.AddExtensionToCategory(ctx, req)
		assert.Nil(t, out)
		assert.Equal(t, data.ErrUnavailable, err)
	})

	t.Run("category must exist", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)

		ctx := manager.StoreDiscoveryManager(context.Background(), buildTestDiscoveryManager(store))
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		req := makeAddExtensionToCategoryRequest("testExtensionID", "testCategoryID", "testVersionID", &defaultOrder)
		out, err := discoveryService.AddExtensionToCategory(ctx, req)
		assert.Nil(t, out)
		assert.Equal(t, data.ErrNotFound, err)
	})

	t.Run("must be authorized", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)

		ctx := manager.StoreDiscoveryManager(context.Background(), buildTestDiscoveryManager(store))
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.NoRole})

		cat, err := store.CreateCategory(&discovery.AddCategoryRequest{
			Category: &discovery.CategoryUpsert{
				Type:    twirputils.WrapString(model.CategoryTypeCurated),
				SortKey: twirputils.WrapString(model.CategorySortkeyManual),
				Visible: twirputils.WrapBool(true),
			},
		})
		require.NoError(t, err)

		req := makeAddExtensionToCategoryRequest("testExtensionID", cat.ID, "testVersionID", &defaultOrder)
		out, err := discoveryService.AddExtensionToCategory(ctx, req)

		assert.Nil(t, out)
		assert.Equal(t, data.ErrUnauthorized, err)
	})

	t.Run("pseudo category", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)

		ctx := manager.StoreDiscoveryManager(context.Background(), buildTestDiscoveryManager(store))
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		cat, err := store.CreateCategory(&discovery.AddCategoryRequest{
			Category: &discovery.CategoryUpsert{
				Type:     twirputils.WrapString(model.CategoryTypePseudo),
				SortKey:  twirputils.WrapString(model.CategorySortkeyManual),
				Visible:  twirputils.WrapBool(false),
				Readonly: twirputils.WrapBool(true),
				Order:    twirputils.WrapFloat(0),
				Slug:     twirputils.WrapString("aslug"),
			},
		})
		require.NoError(t, err)

		req := makeAddExtensionToCategoryRequest(testExtensionID, cat.ID, testVersionID, &defaultOrder)
		req.CategoryID = cat.ID
		out, err := discoveryService.AddExtensionToCategory(ctx, req)
		assert.Nil(t, out)
		assert.Equal(t, data.ErrInvalidCategoryUpdate, err)
	})

	t.Run("dev category should be rejected", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := manager.StoreDiscoveryManager(context.Background(), buildTestDiscoveryManager(store))
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Seer | auth.Categorizer})

		cat, err := store.CreateCategory(&discovery.AddCategoryRequest{
			Category: &discovery.CategoryUpsert{
				Type:    twirputils.WrapString(model.CategoryTypeDeveloper),
				SortKey: twirputils.WrapString(model.CategorySortkeyManual),
			},
		})
		require.NoError(t, err)

		req := makeAddExtensionToCategoryRequest("testExtensionID", cat.ID, "testVersionID", &defaultOrder)
		resp, err := discoveryService.AddExtensionToCategory(ctx, req)
		assert.Nil(t, resp)
		assert.Equal(t, data.ErrInvalidCategoryUpdate, err)
	})

	t.Run("success", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := manager.StoreDiscoveryManager(context.Background(), buildTestDiscoveryManager(store))
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Seer | auth.Categorizer})

		cats := []*model.Category{}
		for i := 0; i < 2; i++ {
			cat, err := store.CreateCategory(&discovery.AddCategoryRequest{
				Category: &discovery.CategoryUpsert{
					Type:    twirputils.WrapString(model.CategoryTypeCurated),
					SortKey: twirputils.WrapString(model.CategorySortkeyManual),
				},
			})
			require.NoError(t, err)
			require.NotNil(t, cat)
			cats = append(cats, cat)
		}

		req := makeAddExtensionToCategoryRequest("testExtensionID", cats[0].ID, "testVersionID", nil)
		resp, err := discoveryService.AddExtensionToCategory(ctx, req)
		require.Nil(t, err)
		assert.Equal(t, cats[0].ID, resp.GetExtensionCategoryMembership().GetCategoryID())
		assert.NotNil(t, resp.GetExtensionCategoryMembership().GetOrder())
		assert.Equal(t, float32(0), resp.GetExtensionCategoryMembership().GetOrder().Value)

		t.Run("order increases", func(t *testing.T) {
			req := makeAddExtensionToCategoryRequest("testExtensionID_2", cats[0].ID, "testVersionID", nil)
			resp, err := discoveryService.AddExtensionToCategory(ctx, req)
			require.Nil(t, err)
			assert.Equal(t, cats[0].ID, resp.GetExtensionCategoryMembership().GetCategoryID())
			assert.NotNil(t, resp.GetExtensionCategoryMembership().GetOrder())
			assert.Equal(t, float32(1), resp.GetExtensionCategoryMembership().GetOrder().Value)

			req = makeAddExtensionToCategoryRequest("testExtensionID_3", cats[0].ID, "testVersionID", nil)
			resp, err = discoveryService.AddExtensionToCategory(ctx, req)
			require.Nil(t, err)
			assert.Equal(t, cats[0].ID, resp.GetExtensionCategoryMembership().GetCategoryID())
			assert.NotNil(t, resp.GetExtensionCategoryMembership().GetOrder())
			assert.Equal(t, float32(2), resp.GetExtensionCategoryMembership().GetOrder().Value)
		})

		t.Run("order stays independent on a different category", func(t *testing.T) {
			req := makeAddExtensionToCategoryRequest("testExtensionID", cats[1].ID, "testVersionID", nil)
			resp, err := discoveryService.AddExtensionToCategory(ctx, req)
			require.Nil(t, err)
			assert.Equal(t, cats[1].ID, resp.GetExtensionCategoryMembership().GetCategoryID())
			assert.NotNil(t, resp.GetExtensionCategoryMembership().GetOrder())
			assert.Equal(t, float32(0), resp.GetExtensionCategoryMembership().GetOrder().Value)

			req = makeAddExtensionToCategoryRequest("testExtensionID_2", cats[1].ID, "testVersionID", nil)
			resp, err = discoveryService.AddExtensionToCategory(ctx, req)
			require.Nil(t, err)
			assert.Equal(t, cats[1].ID, resp.GetExtensionCategoryMembership().GetCategoryID())
			assert.NotNil(t, resp.GetExtensionCategoryMembership().GetOrder())
			assert.Equal(t, float32(1), resp.GetExtensionCategoryMembership().GetOrder().Value)
		})
	})
}

func makeAddExtensionToCategoryRequest(extensionID, categoryID, versionID string, order *float32) *discovery.AddExtensionToCategoryRequest {
	if order == nil {
		return &discovery.AddExtensionToCategoryRequest{
			ExtensionID: extensionID,
			CategoryID:  categoryID,
			VersionID:   twirputils.WrapString(versionID),
		}
	}

	return &discovery.AddExtensionToCategoryRequest{
		Order:       twirputils.WrapFloat(*order),
		ExtensionID: extensionID,
		CategoryID:  categoryID,
		VersionID:   twirputils.WrapString(versionID),
	}

}
