package main

import (
	"context"
	"testing"

	"code.justin.tv/extensions/discovery/auth"
	"code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/discovery/data"
	"code.justin.tv/extensions/discovery/data/model/memory"
	"code.justin.tv/extensions/discovery/golibs/uuid"
	"code.justin.tv/extensions/discovery/manager"
	"code.justin.tv/extensions/discovery/twirputils"

	"github.com/stretchr/testify/assert"
)

func TestAddFeaturedCarouselEntry(t *testing.T) {
	t.Run("Must have a request", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)

		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))

		out, err := discoveryService.AddFeaturedCarouselEntry(ctx, nil)
		assert.Nil(t, out)
		assert.Equal(t, data.ErrMissingParameter, err)
	})

	t.Run("Manager must be set on context", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()

		out, err := discoveryService.AddFeaturedCarouselEntry(ctx, makeAddFeaturedCarouselEntryRequest("SOME_ID"))
		assert.Nil(t, out)
		assert.Equal(t, data.ErrUnavailable, err)
	})

	t.Run("Must be allowed to edit categories", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))

		out, err := discoveryService.AddFeaturedCarouselEntry(ctx, makeAddFeaturedCarouselEntryRequest("SOME_ID"))
		assert.Nil(t, out)
		assert.Equal(t, data.ErrUnauthorized, err)
	})

	t.Run("Can create an entry successfully", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		out, err := discoveryService.AddFeaturedCarouselEntry(ctx, makeAddFeaturedCarouselEntryRequest("SOME_ID"))
		assert.NotNil(t, out)
		assert.Nil(t, err)
	})
}

func makeAddFeaturedCarouselEntryRequest(carouselID string) *discovery.AddFeaturedCarouselEntryRequest {
	return &discovery.AddFeaturedCarouselEntryRequest{
		FeaturedCarouselEntry: &discovery.FeaturedCarouselEntryUpsert{
			CarouselID: twirputils.WrapString(carouselID),
			Content: &discovery.FeaturedCarouselEntryContent{
				Title:        "title",
				ClickThrough: "clickthrough",
				ImageURL:     "imageurl",
			},
		},
	}
}
