package main

import (
	"context"
	"testing"
	"time"

	"code.justin.tv/extensions/discovery/auth"
	"code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/discovery/data"
	"code.justin.tv/extensions/discovery/data/model/memory"
	"code.justin.tv/extensions/discovery/golibs/uuid"
	"code.justin.tv/extensions/discovery/manager"
	"code.justin.tv/extensions/discovery/twirputils"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestAddFeaturedCarousel(t *testing.T) {

	t.Run("Must have a request", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()

		out, err := discoveryService.AddFeaturedCarousel(ctx, nil)
		assert.Nil(t, out)
		assert.Equal(t, data.ErrMissingParameter, err)
	})

	t.Run("Must have timestamp", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		req := makeAddFeaturedCarouselRequest(t, "123")
		req.FeaturedCarousel.StartTime = nil
		out, err := discoveryService.AddFeaturedCarousel(ctx, req)
		assert.Nil(t, out)
		assert.Equal(t, data.ErrMissingParameter, err)
	})

	t.Run("Manager must be set on context", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()

		out, err := discoveryService.AddFeaturedCarousel(ctx, makeAddFeaturedCarouselRequest(t, "123"))
		assert.Nil(t, out)
		assert.Equal(t, data.ErrUnavailable, err)
	})

	t.Run("Must be allowed to edit categories", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))

		out, err := discoveryService.AddFeaturedCarousel(ctx, makeAddFeaturedCarouselRequest(t, "123"))
		assert.Nil(t, out)
		assert.Equal(t, data.ErrUnauthorized, err)
	})

	t.Run("Can create a carousel successfully", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		out, err := discoveryService.AddFeaturedCarousel(ctx, makeAddFeaturedCarouselRequest(t, "123"))
		assert.NotNil(t, out)
		assert.Nil(t, err)
	})
}

func makeAddFeaturedCarouselRequest(t *testing.T, scheduleID string) *discovery.AddFeaturedCarouselRequest {
	now := time.Now()
	nowTwirp, err := twirputils.WrapTimePointer(&now)
	require.Nil(t, err, "Must not have error creating protobuff timestamp")

	return &discovery.AddFeaturedCarouselRequest{
		FeaturedCarousel: &discovery.FeaturedCarouselUpsert{
			ScheduleID: scheduleID,
			StartTime:  nowTwirp,
		},
	}
}
