package main

import (
	"context"
	"testing"

	"code.justin.tv/extensions/discovery/auth"
	"code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/discovery/data"
	"code.justin.tv/extensions/discovery/data/model/memory"
	"code.justin.tv/extensions/discovery/golibs/uuid"
	"code.justin.tv/extensions/discovery/manager"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestAddGameToExtension(t *testing.T) {
	t.Run("must have request", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))

		out, err := discoveryService.AddGameToExtension(ctx, nil)
		assert.Nil(t, out)
		assert.Equal(t, data.ErrMissingParameter, err)
	})

	t.Run("must have maanger", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()

		out, err := discoveryService.AddGameToExtension(ctx, makeAddGameToExtensionRequest())
		assert.Nil(t, out)
		assert.Equal(t, data.ErrUnavailable, err)
	})

	t.Run("discovery data must exist", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		out, err := discoveryService.AddGameToExtension(ctx, makeParameterAddGameToExtensionRequest("dne", "dne", 1))
		assert.Nil(t, out)
		assert.Equal(t, data.ErrNotFound, err)
	})

	t.Run("success", func(t *testing.T) {
		t.Run("curators are allowed to add games to released extensions", func(t *testing.T) {
			store := memory.New(uuid.NewSource())
			discoveryService := fakeServer(t, store)
			ctx := context.Background()
			ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
			ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Seer | auth.Categorizer})

			gameID := int32(1)

			discoData, err := store.PutDiscoveryData(&discovery.PutExtensionVersionDiscoveryDataRequest{
				UpdateData: &discovery.ExtensionVersionDiscoveryDataUpdate{
					ExtensionID: testExtensionID,
					Version:     testVersionID,
				},
			})
			require.NoError(t, err)

			out, err := discoveryService.AddGameToExtension(ctx, makeParameterAddGameToExtensionRequest(discoData.ExtensionID, discoData.Version, gameID))
			assert.Nil(t, err)
			require.NotNil(t, out)
			assert.Equal(t, gameID, out.GetExtensionGameMembership().GetGameID())
			assert.Equal(t, testExtensionID, out.GetExtensionGameMembership().GetExtensionID())
		})
	})
}

func makeAddGameToExtensionRequest() *discovery.AddGameToExtensionRequest {
	return makeParameterAddGameToExtensionRequest("testExtensionID", "testVersionID", 1)
}

func makeParameterAddGameToExtensionRequest(extensionID, versionID string, gameID int32) *discovery.AddGameToExtensionRequest {
	return &discovery.AddGameToExtensionRequest{
		ExtensionID: extensionID,
		VersionID:   versionID,
		GameID:      gameID,
	}
}
