package main

import (
	"context"
	"testing"

	"code.justin.tv/extensions/discovery/auth"
	"code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/discovery/data"
	"code.justin.tv/extensions/discovery/data/model"
	"code.justin.tv/extensions/discovery/data/model/memory"
	"code.justin.tv/extensions/discovery/golibs/uuid"
	"code.justin.tv/extensions/discovery/manager"
	"code.justin.tv/extensions/discovery/twirputils"

	"github.com/stretchr/testify/assert"
)

func TestDeleteCategory(t *testing.T) {
	t.Run("Happy case", func(t *testing.T) {
		t.Run("Delete curated category", func(t *testing.T) {
			store := memory.New(uuid.NewSource())
			discoveryService := fakeServer(t, store)
			m := buildTestDiscoveryManager(store)
			ctx := context.Background()
			ctx = manager.StoreDiscoveryManager(ctx, m)
			ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

			req := validAddCategoryRequest()
			req.Category.Type = twirputils.WrapString(model.CategoryTypeCurated)
			req.Category.SortKey = twirputils.WrapString("manual")
			cat := createCategoryWithNoErrors(t, discoveryService, m, req)

			out, err := discoveryService.DeleteCategory(ctx, makeDeleteCategoryRequest(cat.GetID()))
			assert.Nil(t, err)
			assert.NotNil(t, out)
		})

		t.Run("Delete developer category", func(t *testing.T) {
			store := memory.New(uuid.NewSource())
			discoveryService := fakeServer(t, store)
			m := buildTestDiscoveryManager(store)
			ctx := context.Background()
			ctx = manager.StoreDiscoveryManager(ctx, m)
			ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

			req := validAddCategoryRequest()
			req.Category.Type = twirputils.WrapString(model.CategoryTypeDeveloper)
			req.Category.SortKey = twirputils.WrapString("manual")
			req.Category.Readonly = twirputils.WrapBool(true)
			req.Category.Visible = twirputils.WrapBool(false)
			cat := createCategoryWithNoErrors(t, discoveryService, m, req)

			out, err := discoveryService.DeleteCategory(ctx, makeDeleteCategoryRequest(cat.GetID()))
			assert.Nil(t, err)
			assert.NotNil(t, out)
			assert.True(t, out.GetCategory().GetDeleted())
		})
	})

	t.Run("Sad Cases", func(t *testing.T) {
		t.Run("Request must exist", func(t *testing.T) {
			store := memory.New(uuid.NewSource())
			discoveryService := fakeServer(t, store)
			ctx := context.Background()
			ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
			ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

			out, err := discoveryService.DeleteCategory(ctx, nil)
			assert.Nil(t, out)
			assert.NotNil(t, err)
			assert.Equal(t, data.ErrMissingParameter, err)
		})

		t.Run("Request must have ID", func(t *testing.T) {
			store := memory.New(uuid.NewSource())
			discoveryService := fakeServer(t, store)
			ctx := context.Background()
			ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
			ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

			out, err := discoveryService.DeleteCategory(ctx, &discovery.DeleteCategoryRequest{})
			assert.Nil(t, out)
			assert.NotNil(t, err)
			assert.Equal(t, data.ErrMissingParameter, err)
		})

		t.Run("Store must be set", func(t *testing.T) {
			store := memory.New(uuid.NewSource())
			discoveryService := fakeServer(t, store)
			ctx := context.Background()
			ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

			out, err := discoveryService.DeleteCategory(ctx, makeDeleteCategoryRequest("someid"))
			assert.Nil(t, out)
			assert.NotNil(t, err)
			assert.Equal(t, data.ErrUnavailable, err)
		})

		t.Run("Must be able to categorize", func(t *testing.T) {
			store := memory.New(uuid.NewSource())
			discoveryService := fakeServer(t, store)
			ctx := context.Background()
			ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))

			out, err := discoveryService.DeleteCategory(ctx, makeDeleteCategoryRequest("someid"))
			assert.Nil(t, out)
			assert.NotNil(t, err)
			assert.Equal(t, data.ErrUnauthorized, err)
		})

		t.Run("Category doesn't exist", func(t *testing.T) {
			store := memory.New(uuid.NewSource())
			discoveryService := fakeServer(t, store)
			ctx := context.Background()
			ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
			ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

			out, err := discoveryService.DeleteCategory(ctx, makeDeleteCategoryRequest("someid"))
			assert.Nil(t, out)
			assert.NotNil(t, err)
			assert.Equal(t, data.ErrNotFound, err)
		})

		t.Run("Category isn't readonly", func(t *testing.T) {
			store := memory.New(uuid.NewSource())
			discoveryService := fakeServer(t, store)
			m := buildTestDiscoveryManager(store)
			ctx := context.Background()
			ctx = manager.StoreDiscoveryManager(ctx, m)
			ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

			req := validAddCategoryRequest()
			req.Category.Type = twirputils.WrapString(model.CategoryTypeDeveloper)
			req.Category.SortKey = twirputils.WrapString("manual")
			req.Category.Readonly = twirputils.WrapBool(false)
			cat := createCategoryWithNoErrors(t, discoveryService, m, req)

			out, err := discoveryService.DeleteCategory(ctx, makeDeleteCategoryRequest(cat.GetID()))
			assert.Nil(t, out)
			assert.NotNil(t, err)
			assert.Equal(t, data.ErrNotReadOnly, err)
		})

		t.Run("Category is visible", func(t *testing.T) {
			store := memory.New(uuid.NewSource())
			discoveryService := fakeServer(t, store)
			m := buildTestDiscoveryManager(store)
			ctx := context.Background()
			ctx = manager.StoreDiscoveryManager(ctx, m)
			ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

			req := validAddCategoryRequest()
			req.Category.Type = twirputils.WrapString(model.CategoryTypeDeveloper)
			req.Category.SortKey = twirputils.WrapString("manual")
			req.Category.Readonly = twirputils.WrapBool(true)
			req.Category.Visible = twirputils.WrapBool(true)
			cat := createCategoryWithNoErrors(t, discoveryService, m, req)

			out, err := discoveryService.DeleteCategory(ctx, makeDeleteCategoryRequest(cat.GetID()))
			assert.Nil(t, out)
			assert.NotNil(t, err)
			assert.Equal(t, data.ErrCategoryVisible, err)
		})

		t.Run("Category has associated extensions still", func(t *testing.T) {
			store := memory.New(uuid.NewSource())
			discoveryService := fakeServer(t, store)
			m := buildTestDiscoveryManager(store)
			ctx := context.Background()
			ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
			ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

			req := validAddCategoryRequest()
			req.Category.Type = twirputils.WrapString(model.CategoryTypeDeveloper)
			req.Category.SortKey = twirputils.WrapString("manual")
			req.Category.Readonly = twirputils.WrapBool(true)
			req.Category.Visible = twirputils.WrapBool(false)
			cat := createCategoryWithNoErrors(t, discoveryService, m, req)

			addExtensionToCategory(t, store, cat.GetID(), testExtensionID, testVersionID)

			out, err := discoveryService.DeleteCategory(ctx, makeDeleteCategoryRequest(cat.GetID()))

			assert.Nil(t, out)
			assert.NotNil(t, err)
			assert.Equal(t, data.ErrNonEmptyCategory, err)
		})
	})
}

func makeDeleteCategoryRequest(categoryID string) *discovery.DeleteCategoryRequest {
	return &discovery.DeleteCategoryRequest{
		CategoryID: categoryID,
	}
}
