package main

import (
	"context"
	"testing"

	"code.justin.tv/extensions/discovery/auth"
	"code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/discovery/data"
	"code.justin.tv/extensions/discovery/data/model/memory"
	"code.justin.tv/extensions/discovery/golibs/uuid"
	"code.justin.tv/extensions/discovery/manager"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestDeleteExtensionVersion(t *testing.T) {
	t.Run("Happy case", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		m := buildTestDiscoveryManager(store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, m)
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		discoData, err := store.PutDiscoveryData(&discovery.PutExtensionVersionDiscoveryDataRequest{
			UpdateData: &discovery.ExtensionVersionDiscoveryDataUpdate{
				ExtensionID: testExtensionID,
				Version:     testVersionID,
			},
		})
		require.NoError(t, err)

		out, err := discoveryService.DeleteExtensionVersionDiscoveryData(ctx, makeDeleteExtensionVersionRequest(discoData.ExtensionID, discoData.Version))
		assert.Nil(t, err)
		assert.NotNil(t, out)

		t.Run("verify soft delete", func(t *testing.T) {
			t.Run("public should not see", func(t *testing.T) {
				ctx := context.Background()
				ctx = manager.StoreDiscoveryManager(ctx, m)

				out, err := discoveryService.GetExtensionVersionDiscoveryData(ctx, makeGetExtensionVersionDiscoveryDataRequest(discoData.ExtensionID, discoData.Version))
				assert.Nil(t, out)
				assert.Equal(t, data.ErrNotFound, err)
			})
			t.Run("categorizer should see", func(t *testing.T) {
				ctx := context.Background()
				ctx = manager.StoreDiscoveryManager(ctx, m)
				ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

				out, err := discoveryService.GetExtensionVersionDiscoveryData(ctx, makeGetExtensionVersionDiscoveryDataRequest(discoData.ExtensionID, discoData.Version))
				require.NotNil(t, out)
				assert.Equal(t, testExtensionID, out.GetExtensionVersionData().GetExtensionID())
				assert.Equal(t, testVersionID, out.GetExtensionVersionData().GetVersion())
				assert.Nil(t, err)
			})
		})
	})

	t.Run("Sad Cases", func(t *testing.T) {
		t.Run("Request must exist", func(t *testing.T) {
			store := memory.New(uuid.NewSource())
			discoveryService := fakeServer(t, store)
			ctx := context.Background()
			ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
			ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

			out, err := discoveryService.DeleteExtensionVersionDiscoveryData(ctx, nil)
			assert.Nil(t, out)
			assert.NotNil(t, err)
			assert.Equal(t, data.ErrMissingParameter, err)
		})

		t.Run("Request must have IDs", func(t *testing.T) {
			store := memory.New(uuid.NewSource())
			discoveryService := fakeServer(t, store)
			ctx := context.Background()
			ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
			ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

			out, err := discoveryService.DeleteExtensionVersionDiscoveryData(ctx, &discovery.DeleteExtensionVersionDiscoveryDataRequest{})
			assert.Nil(t, out)
			assert.NotNil(t, err)
			assert.Equal(t, data.ErrMissingParameter, err)

			out, err = discoveryService.DeleteExtensionVersionDiscoveryData(ctx, &discovery.DeleteExtensionVersionDiscoveryDataRequest{
				ExtensionID: testExtensionID,
			})
			assert.Nil(t, out)
			assert.NotNil(t, err)
			assert.Equal(t, data.ErrMissingParameter, err)

			out, err = discoveryService.DeleteExtensionVersionDiscoveryData(ctx, &discovery.DeleteExtensionVersionDiscoveryDataRequest{
				Version: testVersionID,
			})
			assert.Nil(t, out)
			assert.NotNil(t, err)
			assert.Equal(t, data.ErrMissingParameter, err)
		})

		t.Run("Store must be set", func(t *testing.T) {
			store := memory.New(uuid.NewSource())
			discoveryService := fakeServer(t, store)
			ctx := context.Background()
			ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

			out, err := discoveryService.DeleteExtensionVersionDiscoveryData(ctx, makeDeleteExtensionVersionRequest(testExtensionID, testVersionID))
			assert.Nil(t, out)
			assert.NotNil(t, err)
			assert.Equal(t, data.ErrUnavailable, err)
		})

		t.Run("Must be able to categorize", func(t *testing.T) {
			store := memory.New(uuid.NewSource())
			discoveryService := fakeServer(t, store)
			ctx := context.Background()
			ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))

			out, err := discoveryService.DeleteExtensionVersionDiscoveryData(ctx, makeDeleteExtensionVersionRequest(testExtensionID, testVersionID))
			assert.Nil(t, out)
			assert.NotNil(t, err)
			assert.Equal(t, data.ErrUnauthorized, err)
		})

		t.Run("Entry doesn't exist", func(t *testing.T) {
			store := memory.New(uuid.NewSource())
			discoveryService := fakeServer(t, store)
			ctx := context.Background()
			ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
			ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

			out, err := discoveryService.DeleteExtensionVersionDiscoveryData(ctx, makeDeleteExtensionVersionRequest(testExtensionID, testVersionID))
			assert.Nil(t, out)
			assert.NotNil(t, err)
			assert.Equal(t, data.ErrNotFound, err)
		})
	})
}

func makeDeleteExtensionVersionRequest(extensionID, versionID string) *discovery.DeleteExtensionVersionDiscoveryDataRequest {
	return &discovery.DeleteExtensionVersionDiscoveryDataRequest{
		ExtensionID: extensionID,
		Version:     versionID,
	}
}
