package main

import (
	"context"
	"testing"

	"code.justin.tv/extensions/discovery/auth"
	"code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/discovery/data"
	"code.justin.tv/extensions/discovery/data/model/memory"
	"code.justin.tv/extensions/discovery/golibs/uuid"
	"code.justin.tv/extensions/discovery/manager"

	"github.com/stretchr/testify/assert"
)

func TestDeleteFeaturedCarouselEntry(t *testing.T) {
	t.Run("Happy case", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		m := buildTestDiscoveryManager(store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, m)
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		carousel := createFeaturedCarousel(t, discoveryService, m, "123")
		entry := createFeaturedCarouselEntry(t, discoveryService, m, carousel.GetID())

		out, err := discoveryService.DeleteFeaturedCarouselEntry(ctx, makeDeleteFeaturedCarouselEntryRequest(entry.GetID()))
		assert.Nil(t, err)
		assert.NotNil(t, out)
	})

	t.Run("Sad Cases", func(t *testing.T) {
		t.Run("Request must exist", func(t *testing.T) {
			store := memory.New(uuid.NewSource())
			discoveryService := fakeServer(t, store)
			ctx := context.Background()
			ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
			ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

			out, err := discoveryService.DeleteFeaturedCarouselEntry(ctx, nil)
			assert.Nil(t, out)
			assert.NotNil(t, err)
			assert.Equal(t, data.ErrMissingParameter, err)
		})

		t.Run("Request must have ID", func(t *testing.T) {
			store := memory.New(uuid.NewSource())
			discoveryService := fakeServer(t, store)
			ctx := context.Background()
			ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
			ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

			out, err := discoveryService.DeleteFeaturedCarouselEntry(ctx, &discovery.DeleteFeaturedCarouselEntryRequest{})
			assert.Nil(t, out)
			assert.NotNil(t, err)
			assert.Equal(t, data.ErrMissingParameter, err)
		})

		t.Run("Store must be set", func(t *testing.T) {
			store := memory.New(uuid.NewSource())
			discoveryService := fakeServer(t, store)
			ctx := context.Background()
			ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

			out, err := discoveryService.DeleteFeaturedCarouselEntry(ctx, makeDeleteFeaturedCarouselEntryRequest("someid"))
			assert.Nil(t, out)
			assert.NotNil(t, err)
			assert.Equal(t, data.ErrUnavailable, err)
		})

		t.Run("Must be able to categorize", func(t *testing.T) {
			store := memory.New(uuid.NewSource())
			discoveryService := fakeServer(t, store)
			ctx := context.Background()
			ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))

			out, err := discoveryService.DeleteFeaturedCarouselEntry(ctx, makeDeleteFeaturedCarouselEntryRequest("someid"))
			assert.Nil(t, out)
			assert.NotNil(t, err)
			assert.Equal(t, data.ErrUnauthorized, err)
		})

		t.Run("Entry doesn't exist", func(t *testing.T) {
			store := memory.New(uuid.NewSource())
			discoveryService := fakeServer(t, store)
			ctx := context.Background()
			ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
			ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

			out, err := discoveryService.DeleteFeaturedCarouselEntry(ctx, makeDeleteFeaturedCarouselEntryRequest("someid"))
			assert.Nil(t, out)
			assert.NotNil(t, err)
			assert.Equal(t, data.ErrNotFound, err)
		})
	})
}

func makeDeleteFeaturedCarouselEntryRequest(entryID string) *discovery.DeleteFeaturedCarouselEntryRequest {
	return &discovery.DeleteFeaturedCarouselEntryRequest{
		ID: entryID,
	}
}
