package main

import (
	"context"
	"testing"

	"code.justin.tv/extensions/discovery/auth"
	"code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/discovery/data"
	"code.justin.tv/extensions/discovery/data/model/memory"
	"code.justin.tv/extensions/discovery/golibs/uuid"
	"code.justin.tv/extensions/discovery/manager"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestDeleteGameFromExtension(t *testing.T) {
	t.Run("must have request", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		out, err := discoveryService.DeleteGameFromExtension(ctx, nil)
		assert.Nil(t, out)
		assert.Equal(t, data.ErrMissingParameter, err)
	})

	t.Run("must have manager", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		out, err := discoveryService.DeleteGameFromExtension(ctx, makeDeleteGameFromExtensionRequest(testExtensionID, testVersionID, 1))
		assert.Nil(t, out)
		assert.Equal(t, data.ErrUnavailable, err)
	})

	t.Run("extension must exist", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		out, err := discoveryService.DeleteGameFromExtension(ctx, makeDeleteGameFromExtensionRequest(testExtensionID, testVersionID, 1))
		assert.Nil(t, out)
		assert.Equal(t, data.ErrNotFound, err)
	})

	t.Run("must be authorized", func(t *testing.T) {
		t.Run("auth check", func(t *testing.T) {
			store := memory.New(uuid.NewSource())
			discoveryService := fakeServer(t, store)
			ctx := context.Background()
			ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))

			out, err := discoveryService.DeleteGameFromExtension(ctx, makeDeleteGameFromExtensionRequest(testExtensionID, testVersionID, 1))
			assert.Nil(t, out)
			assert.Equal(t, data.ErrUnauthorized, err)
		})
	})

	t.Run("delete game that is not added", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Seer | auth.Categorizer})

		gameID := int32(1)

		discoData, err := store.PutDiscoveryData(&discovery.PutExtensionVersionDiscoveryDataRequest{
			UpdateData: &discovery.ExtensionVersionDiscoveryDataUpdate{
				ExtensionID: testExtensionID,
				Version:     testVersionID,
				Games: &discovery.OptionalIntList{
					Values: []int64{int64(gameID)},
				},
			},
		})
		require.NoError(t, err)

		out, err := discoveryService.DeleteGameFromExtension(ctx, makeDeleteGameFromExtensionRequest(discoData.ExtensionID, discoData.Version, gameID+1))
		assert.Nil(t, out)
		assert.Equal(t, data.ErrGameNotFound, err)
	})

	t.Run("success", func(t *testing.T) {
		t.Run("curators are allowed to delete games to released extensions", func(t *testing.T) {
			store := memory.New(uuid.NewSource())
			discoveryService := fakeServer(t, store)
			ctx := context.Background()
			ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
			ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Seer | auth.Categorizer})

			gameID := int32(1)

			discoData, err := store.PutDiscoveryData(&discovery.PutExtensionVersionDiscoveryDataRequest{
				UpdateData: &discovery.ExtensionVersionDiscoveryDataUpdate{
					ExtensionID: testExtensionID,
					Version:     testVersionID,
					Games: &discovery.OptionalIntList{
						Values: []int64{int64(gameID)},
					},
				},
			})
			require.NoError(t, err)

			out, err := discoveryService.DeleteGameFromExtension(ctx, makeDeleteGameFromExtensionRequest(discoData.ExtensionID, discoData.Version, gameID))
			assert.Nil(t, err)
			require.NotNil(t, out)
			assert.Equal(t, int32(0), out.GetExtensionGameMembership().GetGameID())
			assert.Equal(t, testExtensionID, out.GetExtensionGameMembership().GetExtensionID())
		})
	})

}

func makeDeleteGameFromExtensionRequest(extensionID string, versionID string, gameID int32) *discovery.DeleteGameFromExtensionRequest {
	return &discovery.DeleteGameFromExtensionRequest{
		ExtensionID: extensionID,
		VersionID:   versionID,
		GameID:      gameID,
	}
}
