package main

import (
	"context"
	"testing"

	"code.justin.tv/extensions/discovery/auth"
	"code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/discovery/data"
	"code.justin.tv/extensions/discovery/data/model"
	"code.justin.tv/extensions/discovery/data/model/memory"
	"code.justin.tv/extensions/discovery/golibs/uuid"
	"code.justin.tv/extensions/discovery/manager"
	"code.justin.tv/extensions/discovery/twirputils"

	"github.com/stretchr/testify/assert"
)

func TestEditCategoryTranslations(t *testing.T) {
	t.Run("Happy case", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		m := buildTestDiscoveryManager(store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, m)
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		req := validAddCategoryRequest()
		req.Category.Type = twirputils.WrapString(model.CategoryTypeCurated)
		req.Category.SortKey = twirputils.WrapString("manual")
		cat := createCategoryWithNoErrors(t, discoveryService, m, req)

		out, err := discoveryService.EditCategoryTranslation(ctx,
			makeEditCategoryTranslationsRequest(cat.GetID(), "new1", "de", "newname"))
		assert.Nil(t, err)
		assert.NotNil(t, out)

	})

	t.Run("Sad Cases", func(t *testing.T) {
		t.Run("Request must exist", func(t *testing.T) {
			store := memory.New(uuid.NewSource())
			discoveryService := fakeServer(t, store)
			ctx := context.Background()
			ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
			ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

			out, err := discoveryService.EditCategoryTranslation(ctx, nil)
			assert.Nil(t, out)
			assert.NotNil(t, err)
			assert.Equal(t, data.ErrMissingParameter, err)
		})

		t.Run("Request must have ID", func(t *testing.T) {
			store := memory.New(uuid.NewSource())
			discoveryService := fakeServer(t, store)
			ctx := context.Background()
			ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
			ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

			out, err := discoveryService.EditCategoryTranslation(ctx, &discovery.EditCategoryTranslationRequest{})
			assert.Nil(t, out)
			assert.NotNil(t, err)
			assert.Equal(t, data.ErrMissingParameter, err)
		})

		t.Run("Store must be set", func(t *testing.T) {
			store := memory.New(uuid.NewSource())
			discoveryService := fakeServer(t, store)
			ctx := context.Background()
			ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

			out, err := discoveryService.EditCategoryTranslation(ctx,
				makeEditCategoryTranslationsRequest("someid", "somedesc", "de", "somename"))
			assert.Nil(t, out)
			assert.NotNil(t, err)
			assert.Equal(t, data.ErrUnavailable, err)
		})

		t.Run("Must be able to categorize", func(t *testing.T) {
			store := memory.New(uuid.NewSource())
			discoveryService := fakeServer(t, store)
			ctx := context.Background()
			ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))

			out, err := discoveryService.EditCategoryTranslation(ctx,
				makeEditCategoryTranslationsRequest("someid", "somedesc", "de", "somename"))
			assert.Nil(t, out)
			assert.NotNil(t, err)
			assert.Equal(t, data.ErrUnauthorized, err)
		})

		t.Run("Category doesn't exist", func(t *testing.T) {
			store := memory.New(uuid.NewSource())
			discoveryService := fakeServer(t, store)
			ctx := context.Background()
			ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
			ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

			out, err := discoveryService.EditCategoryTranslation(ctx,
				makeEditCategoryTranslationsRequest("someid", "somedesc", "de", "somename"))
			assert.Nil(t, out)
			assert.NotNil(t, err)
			assert.Equal(t, data.ErrCategoryNotFound, err)
		})
	})
}

func makeEditCategoryTranslationsRequest(categoryID, description, language, name string) *discovery.EditCategoryTranslationRequest {
	return &discovery.EditCategoryTranslationRequest{
		CategoryID:  categoryID,
		Description: description,
		Language:    language,
		Name:        name,
	}
}
