package main

import (
	"context"
	"testing"

	"code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/discovery/data"
	"code.justin.tv/extensions/discovery/data/model/memory"
	"code.justin.tv/extensions/discovery/golibs/uuid"
	"code.justin.tv/extensions/discovery/manager"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestGetCategories(t *testing.T) {
	t.Run("Must have a request", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()

		out, err := discoveryService.GetCategories(ctx, nil)
		assert.Nil(t, out)
		assert.Equal(t, data.ErrMissingParameter, err)
	})

	t.Run("Must have a manager", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()

		out, err := discoveryService.GetCategories(ctx, makeGetCategoriesRequest())
		assert.Nil(t, out)
		assert.Equal(t, data.ErrUnavailable, err)
	})

	t.Run("Can call get with zero categories in store", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))

		out, err := discoveryService.GetCategories(ctx, makeGetCategoriesRequest())
		assert.NotNil(t, out)
		assert.Nil(t, err)
	})

	t.Run("Can call get with a nonempty store", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		m := buildTestDiscoveryManager(store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, m)

		cat := createCategoryWithNoErrors(t, discoveryService, m, validAddCategoryRequest())

		out, err := discoveryService.GetCategories(ctx, makeGetCategoriesRequest())
		assert.Nil(t, err)
		assert.NotNil(t, out)
		assert.Equal(t, int32(1), out.Categories.Count)
		require.Equal(t, 1, len(out.Categories.Categories))
		assert.Equal(t, cat, out.Categories.Categories[0])
	})
}

func makeGetCategoriesRequest() *discovery.GetCategoriesRequest {
	return &discovery.GetCategoriesRequest{
		Limit:  10,
		Offset: 0,
	}
}
