package main

import (
	"context"
	"fmt"
	"testing"

	"code.justin.tv/extensions/discovery/auth"
	discovery "code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/discovery/data"
	"code.justin.tv/extensions/discovery/data/model"
	"code.justin.tv/extensions/discovery/data/model/memory"
	"code.justin.tv/extensions/discovery/golibs/uuid"
	"code.justin.tv/extensions/discovery/manager"
	"code.justin.tv/extensions/discovery/twirputils"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestGetCategoryExtensions(t *testing.T) {
	t.Run("category id is required", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))

		doc, err := discoveryService.GetCategoryExtensions(ctx, &discovery.GetCategoryExtensionsRequest{})
		assert.Nil(t, doc)
		assert.Equal(t, data.ErrMissingParameter, err)
	})

	t.Run("manager must be available in the context", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()

		doc, err := discoveryService.GetCategoryExtensions(ctx, makeGetCategoryExtensionsRequest("123", 0, 5))
		assert.Nil(t, doc)
		assert.Equal(t, data.ErrUnavailable, err)
	})

	t.Run("the category must exist", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))

		doc, err := discoveryService.GetCategoryExtensions(ctx, makeGetCategoryExtensionsRequest("123", 0, 5))
		assert.Nil(t, doc)
		assert.Equal(t, data.ErrNotFound, err)
	})

	t.Run("success", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		m := buildTestDiscoveryManager(store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, m)
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		req := validAddCategoryRequest()
		req.Category.Type = twirputils.WrapString(model.CategoryTypeCurated)
		req.Category.SortKey = twirputils.WrapString(model.CategorySortkeyManual)
		req.Category.Visible = twirputils.WrapBool(true)
		cat := createCategoryWithNoErrors(t, discoveryService, m, req)

		catID := cat.GetID()
		for i := 1; i < 13; i++ {
			addExtensionToCategory(t, store, catID, fmt.Sprintf("%d", i), testVersionID)
		}

		t.Run("returns paginated list of extensions", func(t *testing.T) {
			t.Run("default pagination is limit=10, offset=0", func(t *testing.T) {
				out, err := discoveryService.GetCategoryExtensions(ctx, &discovery.GetCategoryExtensionsRequest{
					CategoryID: catID,
				})

				require.Nil(t, err)
				assert.Equal(t, int32(12), out.CategoryExtensions.TotalCount)
				require.Equal(t, 10, len(out.CategoryExtensions.IDs))
				assert.Equal(t, "1", out.CategoryExtensions.IDs[0])
				assert.Equal(t, "10", out.CategoryExtensions.IDs[9])
			})

			t.Run("respects pagination params", func(t *testing.T) {
				out, err := discoveryService.GetCategoryExtensions(ctx, &discovery.GetCategoryExtensionsRequest{
					CategoryID: catID,
					Offset:     2,
					Limit:      3,
				})

				require.Nil(t, err)
				assert.Equal(t, int32(12), out.CategoryExtensions.TotalCount)
				require.Equal(t, 3, len(out.CategoryExtensions.IDs))
				assert.Equal(t, "3", out.CategoryExtensions.IDs[0])
				assert.Equal(t, "4", out.CategoryExtensions.IDs[1])
				assert.Equal(t, "5", out.CategoryExtensions.IDs[2])
			})
		})
	})
}

func makeGetCategoryExtensionsRequest(categoryID string, offset, limit int32) *discovery.GetCategoryExtensionsRequest {
	return &discovery.GetCategoryExtensionsRequest{
		CategoryID: categoryID,
		Offset:     offset,
		Limit:      limit,
	}
}

func addExtensionToCategory(t *testing.T, store model.Store, categoryID, extensionID, versionID string) {
	addExtReq := makeAddExtensionToCategoryRequest(extensionID, categoryID, versionID, nil)
	addExtReq.CategoryID = categoryID
	addExtReq.ExtensionID = extensionID
	_, err := store.AddExtensionToCategory(addExtReq)
	require.NoError(t, err)
}
