package main

import (
	"context"
	"testing"

	"code.justin.tv/extensions/discovery/auth"
	"code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/discovery/data"
	"code.justin.tv/extensions/discovery/data/model"
	"code.justin.tv/extensions/discovery/data/model/memory"
	"code.justin.tv/extensions/discovery/golibs/uuid"
	"code.justin.tv/extensions/discovery/manager"
	"code.justin.tv/extensions/discovery/twirputils"

	"github.com/stretchr/testify/assert"
)

func TestGetCategory(t *testing.T) {
	t.Run("missing request", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))

		out, err := discoveryService.GetCategory(ctx, nil)
		assert.Nil(t, out)
		assert.Equal(t, data.ErrMissingParameter, err)
	})

	t.Run("must have category ID", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))

		out, err := discoveryService.GetCategory(ctx, &discovery.GetCategoryRequest{})
		assert.Nil(t, out)
		assert.Equal(t, data.ErrMissingParameter, err)
	})

	t.Run("missing store", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = auth.Store(ctx, &auth.FakeCredentials{})

		out, err := discoveryService.GetCategory(ctx, makeGetCategoryRequest("123"))
		assert.Nil(t, out)
		assert.Equal(t, data.ErrUnavailable, err)
	})

	t.Run("category does not exists", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
		ctx = auth.Store(ctx, &auth.FakeCredentials{})

		out, err := discoveryService.GetCategory(ctx, makeGetCategoryRequest("123"))
		assert.Nil(t, out)
		assert.Equal(t, data.ErrNotFound, err)
	})

	t.Run("success", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		m := buildTestDiscoveryManager(store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, m)
		ctx = auth.Store(ctx, &auth.FakeCredentials{})

		req := validAddCategoryRequest()
		req.Category.SortKey = twirputils.WrapString(model.CategorySortkeyManual)
		req.Category.Type = twirputils.WrapString(model.CategoryTypeDeveloper)
		req.Category.Visible = twirputils.WrapBool(true)
		cat := createCategoryWithNoErrors(t, discoveryService, m, req)

		out, err := discoveryService.GetCategory(ctx, makeGetCategoryRequest(cat.GetID()))
		assert.NotNil(t, out)
		assert.Nil(t, err)
	})
}

func makeGetCategoryRequest(categoryID string) *discovery.GetCategoryRequest {
	return &discovery.GetCategoryRequest{
		CategoryID: categoryID,
	}
}
