package main

import (
	"context"
	"testing"

	"code.justin.tv/extensions/discovery/auth"
	"code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/discovery/data"
	"code.justin.tv/extensions/discovery/data/model/memory"
	"code.justin.tv/extensions/discovery/golibs/uuid"
	"code.justin.tv/extensions/discovery/manager"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestGetExtensionVersionDiscoveryData(t *testing.T) {
	t.Run("Must have a request", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))

		out, err := discoveryService.GetExtensionVersionDiscoveryData(ctx, nil)
		assert.Nil(t, out)
		assert.Equal(t, data.ErrMissingParameter, err)
	})

	t.Run("Must have IDs", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))

		out, err := discoveryService.GetExtensionVersionDiscoveryData(ctx, &discovery.GetExtensionVersionDiscoveryDataRequest{})
		assert.Nil(t, out)
		assert.Equal(t, data.ErrMissingParameter, err)

		out, err = discoveryService.GetExtensionVersionDiscoveryData(ctx, &discovery.GetExtensionVersionDiscoveryDataRequest{
			ExtensionID: testExtensionID,
		})
		assert.Nil(t, out)
		assert.Equal(t, data.ErrMissingParameter, err)

		out, err = discoveryService.GetExtensionVersionDiscoveryData(ctx, &discovery.GetExtensionVersionDiscoveryDataRequest{
			Version: testVersionID,
		})
		assert.Nil(t, out)
		assert.Equal(t, data.ErrMissingParameter, err)
	})

	t.Run("Must have a manager", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()

		out, err := discoveryService.GetExtensionVersionDiscoveryData(ctx, makeGetExtensionVersionDiscoveryDataRequest(testExtensionID, testVersionID))
		assert.Nil(t, out)
		assert.Equal(t, data.ErrUnavailable, err)
	})

	t.Run("Can get entry", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		m := buildTestDiscoveryManager(store)
		ctx := context.Background()
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})
		ctx = manager.StoreDiscoveryManager(ctx, m)

		discoData, err := store.PutDiscoveryData(&discovery.PutExtensionVersionDiscoveryDataRequest{
			UpdateData: &discovery.ExtensionVersionDiscoveryDataUpdate{
				ExtensionID: testExtensionID,
				Version:     testVersionID,
			},
		})
		require.NoError(t, err)

		out, err := discoveryService.GetExtensionVersionDiscoveryData(ctx, makeGetExtensionVersionDiscoveryDataRequest(discoData.ExtensionID, discoData.Version))
		assert.NotNil(t, out)
		assert.Nil(t, err)
	})
}

func makeGetExtensionVersionDiscoveryDataRequest(extensionID, versionID string) *discovery.GetExtensionVersionDiscoveryDataRequest {
	return &discovery.GetExtensionVersionDiscoveryDataRequest{
		ExtensionID: extensionID,
		Version:     versionID,
	}
}
