package main

import (
	"context"
	"testing"

	"code.justin.tv/extensions/discovery/auth"
	"code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/discovery/data"
	"code.justin.tv/extensions/discovery/data/model/memory"
	"code.justin.tv/extensions/discovery/golibs/uuid"
	"code.justin.tv/extensions/discovery/manager"

	"github.com/stretchr/testify/assert"
)

func TestGetFeaturedCarouselEntry(t *testing.T) {
	t.Run("Must have a request", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))

		out, err := discoveryService.GetFeaturedCarouselEntry(ctx, nil)
		assert.Nil(t, out)
		assert.Equal(t, data.ErrMissingParameter, err)
	})

	t.Run("Must have an ID", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))

		out, err := discoveryService.GetFeaturedCarouselEntry(ctx, &discovery.GetFeaturedCarouselEntryRequest{})
		assert.Nil(t, out)
		assert.Equal(t, data.ErrMissingParameter, err)
	})

	t.Run("Must have a manager", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()

		out, err := discoveryService.GetFeaturedCarouselEntry(ctx, makeGetFeaturedCarouselEntryRequest("123"))
		assert.Nil(t, out)
		assert.Equal(t, data.ErrUnavailable, err)
	})

	t.Run("Can get entry", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		m := buildTestDiscoveryManager(store)
		ctx := context.Background()
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})
		ctx = manager.StoreDiscoveryManager(ctx, m)

		schedule := createFeaturedSchedule(t, discoveryService, m)
		carousel := createFeaturedCarousel(t, discoveryService, m, schedule.GetID())
		entry := createFeaturedCarouselEntry(t, discoveryService, m, carousel.GetID())

		out, err := discoveryService.GetFeaturedCarouselEntry(ctx, makeGetFeaturedCarouselEntryRequest(entry.GetID()))
		assert.NotNil(t, out)
		assert.Nil(t, err)
	})
}

func makeGetFeaturedCarouselEntryRequest(entryID string) *discovery.GetFeaturedCarouselEntryRequest {
	return &discovery.GetFeaturedCarouselEntryRequest{
		ID: entryID,
	}
}
