package main

import (
	"context"
	"testing"

	"code.justin.tv/extensions/discovery/auth"
	"code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/discovery/data"
	"code.justin.tv/extensions/discovery/data/model/memory"
	"code.justin.tv/extensions/discovery/golibs/uuid"
	"code.justin.tv/extensions/discovery/manager"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestGetFeaturedCarousels(t *testing.T) {
	t.Run("Must have a request", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		out, err := discoveryService.GetFeaturedCarousels(ctx, nil)
		assert.Nil(t, out)
		assert.Equal(t, data.ErrMissingParameter, err)
	})

	t.Run("Must have a manager", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		out, err := discoveryService.GetFeaturedCarousels(ctx, makeGetFeaturedCarouselsRequest())
		assert.Nil(t, out)
		assert.Equal(t, data.ErrUnavailable, err)
	})

	t.Run("Must be authorized", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))

		out, err := discoveryService.GetFeaturedCarousels(ctx, makeGetFeaturedCarouselsRequest())
		assert.Nil(t, out)
		assert.Equal(t, data.ErrUnauthorized, err)
	})

	t.Run("Can call get with zero entries in store", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))

		out, err := discoveryService.GetFeaturedCarousels(ctx, makeGetFeaturedCarouselsRequest())
		assert.NotNil(t, out)
		assert.Nil(t, err)
	})

	t.Run("Can call get with a nonempty store", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		m := buildTestDiscoveryManager(store)
		ctx := context.Background()
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})
		ctx = manager.StoreDiscoveryManager(ctx, m)

		schedule := createFeaturedSchedule(t, discoveryService, m)
		carousel := createFeaturedCarousel(t, discoveryService, m, schedule.GetID())

		out, err := discoveryService.GetFeaturedCarousels(ctx, makeGetFeaturedCarouselsRequest())
		assert.Nil(t, err)
		assert.NotNil(t, out)
		assert.Equal(t, int32(1), out.GetFeaturedCarousels().GetCount())
		require.Equal(t, 1, len(out.GetFeaturedCarousels().GetCarousels()))
		assert.Equal(t, carousel.GetID(), out.GetFeaturedCarousels().GetCarousels()[0].GetID())
	})
}

func makeGetFeaturedCarouselsRequest() *discovery.GetFeaturedCarouselsRequest {
	return &discovery.GetFeaturedCarouselsRequest{
		Offset: 0,
		Limit:  10,
	}
}
