package main

import (
	"context"
	"testing"

	"code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/discovery/data"
	"code.justin.tv/extensions/discovery/data/model/memory"
	"code.justin.tv/extensions/discovery/golibs/uuid"
	"code.justin.tv/extensions/discovery/manager"

	"github.com/stretchr/testify/assert"
)

func TestGetFeaturedSchedules(t *testing.T) {
	t.Run("Must have a request", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))

		out, err := discoveryService.GetFeaturedSchedules(ctx, nil)
		assert.Nil(t, out)
		assert.Equal(t, data.ErrMissingParameter, err)
	})

	t.Run("Must have a manager", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()

		out, err := discoveryService.GetFeaturedSchedules(ctx, makeGetFeaturedSchedulesRequest())
		assert.Nil(t, out)
		assert.Equal(t, data.ErrUnavailable, err)
	})

	t.Run("Can call get with zero schedules in store", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))

		out, err := discoveryService.GetFeaturedSchedules(ctx, makeGetFeaturedSchedulesRequest())
		assert.NotNil(t, out)
		assert.Nil(t, err)
	})

	t.Run("Can call get with a nonempty store", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		m := buildTestDiscoveryManager(store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, m)

		schedule := createFeaturedSchedule(t, discoveryService, m)
		_ = createFeaturedCarousel(t, discoveryService, m, schedule.GetID())

		out, err := discoveryService.GetFeaturedSchedules(ctx, makeGetFeaturedSchedulesRequest())
		assert.NotNil(t, out)
		assert.Nil(t, err)
	})
}

func makeGetFeaturedSchedulesRequest() *discovery.GetFeaturedSchedulesRequest {
	return &discovery.GetFeaturedSchedulesRequest{}
}
