package main

import (
	"context"
	"testing"

	"code.justin.tv/extensions/discovery/auth"
	"code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/discovery/data"
	"code.justin.tv/extensions/discovery/data/model/memory"
	"code.justin.tv/extensions/discovery/golibs/uuid"
	"code.justin.tv/extensions/discovery/manager"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestHardDeleteExtension(t *testing.T) {
	t.Run("Happy case", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		m := buildTestDiscoveryManager(store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, m)
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		discoData, err := store.PutDiscoveryData(&discovery.PutExtensionVersionDiscoveryDataRequest{
			UpdateData: &discovery.ExtensionVersionDiscoveryDataUpdate{
				ExtensionID: testExtensionID,
				Version:     testVersionID,
			},
		})
		require.NoError(t, err)

		discoData2, err := store.PutDiscoveryData(&discovery.PutExtensionVersionDiscoveryDataRequest{
			UpdateData: &discovery.ExtensionVersionDiscoveryDataUpdate{
				ExtensionID: testExtensionID,
				Version:     testVersionID + "2",
			},
		})
		require.NoError(t, err)

		assert.Equal(t, discoData.ExtensionID, discoData2.ExtensionID)

		out, err := discoveryService.HardDeleteExtensionDiscoveryData(ctx, makeHardDeleteExtensionRequest(discoData.ExtensionID))
		assert.Nil(t, err)
		assert.NotNil(t, out)
	})

	t.Run("Sad Cases", func(t *testing.T) {
		t.Run("Request must exist", func(t *testing.T) {
			store := memory.New(uuid.NewSource())
			discoveryService := fakeServer(t, store)
			ctx := context.Background()
			ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
			ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

			out, err := discoveryService.HardDeleteExtensionDiscoveryData(ctx, nil)
			assert.Nil(t, out)
			assert.NotNil(t, err)
			assert.Equal(t, data.ErrMissingParameter, err)
		})

		t.Run("Request must have ID", func(t *testing.T) {
			store := memory.New(uuid.NewSource())
			discoveryService := fakeServer(t, store)
			ctx := context.Background()
			ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
			ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

			out, err := discoveryService.HardDeleteExtensionDiscoveryData(ctx, &discovery.HardDeleteExtensionDiscoveryDataRequest{})
			assert.Nil(t, out)
			assert.NotNil(t, err)
			assert.Equal(t, data.ErrMissingParameter, err)
		})

		t.Run("Store must be set", func(t *testing.T) {
			store := memory.New(uuid.NewSource())
			discoveryService := fakeServer(t, store)
			ctx := context.Background()
			ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

			out, err := discoveryService.HardDeleteExtensionDiscoveryData(ctx, makeHardDeleteExtensionRequest(testExtensionID))
			assert.Nil(t, out)
			assert.NotNil(t, err)
			assert.Equal(t, data.ErrUnavailable, err)
		})

		t.Run("Must be able to categorize", func(t *testing.T) {
			store := memory.New(uuid.NewSource())
			discoveryService := fakeServer(t, store)
			ctx := context.Background()
			ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))

			out, err := discoveryService.HardDeleteExtensionDiscoveryData(ctx, makeHardDeleteExtensionRequest(testExtensionID))
			assert.Nil(t, out)
			assert.NotNil(t, err)
			assert.Equal(t, data.ErrUnauthorized, err)
		})

		t.Run("Entry doesn't exist", func(t *testing.T) {
			store := memory.New(uuid.NewSource())
			discoveryService := fakeServer(t, store)
			ctx := context.Background()
			ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
			ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

			out, err := discoveryService.HardDeleteExtensionDiscoveryData(ctx, makeHardDeleteExtensionRequest(testExtensionID))
			assert.Nil(t, out)
			assert.NotNil(t, err)
			assert.Equal(t, data.ErrNotFound, err)
		})
	})
}

func makeHardDeleteExtensionRequest(extensionID string) *discovery.HardDeleteExtensionDiscoveryDataRequest {
	return &discovery.HardDeleteExtensionDiscoveryDataRequest{
		ExtensionID: extensionID,
	}
}
