package main

import (
	"crypto/tls"
	"encoding/json"
	"fmt"
	"net/http"
	"os"

	"code.justin.tv/amzn/TwitchTelemetry"
	"code.justin.tv/extensions/discovery/cmd/discovery/TwitchLoggingMiddleware"
	"code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/discovery/data/model"
	"code.justin.tv/extensions/discovery/data/model/postgres"
	"code.justin.tv/extensions/discovery/fultonlibs/FultonGoLangBootstrap/bootstrap"
	"code.justin.tv/extensions/discovery/fultonlibs/FultonGoLangBootstrap/fultonecs"
	stacklogger "code.justin.tv/extensions/discovery/golibs/logger"
	"code.justin.tv/extensions/discovery/manager"
	"code.justin.tv/security/ephemeralcert"
	"code.justin.tv/video/metricmiddleware-beta/twirpmetric"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/secretsmanager"
	"github.com/pkg/errors"
	"github.com/twitchtv/twirp"

	"gopkg.in/validator.v2"
)

var (
	serviceName = "discovery"
)

type discoveryConfig struct {
	Port          int64 `c7:"port" validate:"nonzero"`
	HTTPPort      int64 `c7:"httpPort" validate:"nonzero"`

	DBSecretName string `c7:"db_secret_name" validate:"nonzero"`
	// DBHostOverride is used when running discovery service locally on a laptop, connecting to the database over a
	// ssh proxy connection, where the database is available on localhost:5432.  You want to set the db_host_override
	// to 'localhost' in this scenario.  This is done in the all.c7 file already so it should "just work" when you
	// run the server using `make test && make local`.
	DBHostOverride string `c7:"db_host_override"`
}

//
// This function adds environment variables that the Fulton bootstrap process requires.
//
// Environment variables that are provided by ECS when it starts the containers up, as we have defined in
// the /buildspec.yml file:
//
//  ENVIRONMENT
//  GIT_COMMIT
//
// Env vars required by the Fulton bootstrap process:
//
//  FULTON_STAGE - matches the ENVIRONMENT var we currently use
//  FULTON_SUBSTAGE - not important, has a default value "primary" if unset
//  AWS_DEFAULT_REGION - region env var provided by ECS
//  ECS_CLUSTER - the name of the cluster running this app, used as a dimension in cloudwatch to identify which cluster
//     originated which metrics
//  HOSTNAME - not important when running in ECS
//  FULTON_VERSION - this can match GIT_COMMIT
//
// AWS_DEFAULT_REGION is required by the cloudwatch sender
// AWS_DEFAULT_REGION and FULTON_STAGE are required by c7 loader to identify the most relevant config params
//
// Environment variables available when running in FARGATE:
//
// HOSTNAME=ip-10-205-12-31.us-west-2.compute.internal
// AWS_CONTAINER_CREDENTIALS_RELATIVE_URI=/v2/credentials/948faa71-3a56-4751-99a6-b1afbe3ae6a6
// ECS_CONTAINER_METADATA_URI=http://169.254.170.2/v3/5fae09b4-af1a-4fcb-b2d4-9c6da59996c9
// AWS_DEFAULT_REGION=us-west-2
// AWS_REGION=us-west-2
// AWS_EXECUTION_ENV=AWS_ECS_FARGATE
// HOME=/root
//
func createFultonEnvironmentVars() {
	env := os.Getenv("ENVIRONMENT")

	if env == "" {
		env = "local"
		_ = os.Setenv("ENVIRONMENT", env)
	}

	gitCommit := os.Getenv("GIT_COMMIT")

	if gitCommit == "" {
		gitCommit = "unknown"
		_ = os.Setenv("GIT_COMMIT", gitCommit)
	}

	_ = os.Setenv("FULTON_STAGE", env)
	_ = os.Setenv("FULTON_VERSION", gitCommit)

	// ECS_CLUSTER just needs to identify the cluster that the metrics were emitted from
	// this is AWS ECS not extensions ECS
	_ = os.Setenv("ECS_CLUSTER", serviceName)
}

//
// I want a logger that:
// has Info, Warn, Error methods that automatically include a dimension of logLevel and set it to those values
// has a Log method that takes a string, an optional map[string]interface{} called dimensions, and an optional TraceChain
// object.
// That prints json to stdout with a minimum format of {"msg":"error message","dimensions":{"level":"info"}}
// TraceChain should be omitempty
// I want this so I can include it in the db.Store and then call it in get_carousel_entries_by_ids.go where we just
// silently ignore getting back a row that we shouldn't.
//
func main() {
	logger := stacklogger.NewJsonLogger(json.NewEncoder(os.Stdout))

	createFultonEnvironmentVars()

	env := os.Getenv("ENVIRONMENT")
	gitCommit := os.Getenv("GIT_COMMIT")

	logger.Info(fmt.Sprintf("Starting server from git commit %s in %s", gitCommit, env), nil, nil)

	// This loads the config from "all.c7"
	bs, err := fultonecs.ECS(serviceName, bootstrap.DefaultConfigSetLoader)
	if err != nil {
		logger.Fatal(fmt.Sprintf("Error bootstrapping: %v", err), nil, nil)
		os.Exit(1)
	}

	bs.SampleReporter.Report("ServerColdStart", 1, telemetry.UnitCount)

	config := discoveryConfig{}

	err = bs.C7.FillWithNamespace(serviceName, &config)
	if err != nil {
		logger.Fatal(fmt.Sprintf("Error loading config: %s", err), nil, nil)
		os.Exit(1)
	}

	err = validator.Validate(config)

	if err != nil {
		logger.Fatal(fmt.Sprintf("Error in configuration: %s", err), nil, nil)
		os.Exit(1)
	}

	metricsMiddleware := &twirpmetric.Server{Starter: bs.OperationStarter}

	errorLoggingMiddleware := logmiddleware.Server{
		Encoder: json.NewEncoder(os.Stdout),

		// Status codes that are just noise and don't need to be logged can be filtered here.
		// Suppressing the logging won't affect metrics.
		// SuppressedErrorCodes: []twirp.ErrorCode{twirp.Unauthenticated, twirp.BadRoute},
	}

	datastore, err := createDiscomanStore(config, logger)
	if err != nil {
		logger.Fatal("Could not create discoman store", nil, stacklogger.GetStackChain(err))
		os.Exit(1)
	}

	managerMiddleware := manager.DiscoveryManagerServerHook{
		Manager: manager.New(datastore),
	}

	twirpMiddleware := twirp.ChainHooks(
		metricsMiddleware.ServerHooks(),
		errorLoggingMiddleware.ServerHooks(),
		managerMiddleware.ServerHooks(),
	)

	discoveryServerImpl, err := NewDiscoveryServer(config, datastore, bs.SampleReporter, logger)

	if err != nil {
		logger.Fatal(fmt.Sprintf("Failed to create twirp server: %s", err), nil, nil)
		os.Exit(1)
	}

	// turn that into an actual twirpServer that can handle http requests
	twirpHandler := discovery.NewDiscoveryServer(discoveryServerImpl, twirpMiddleware)

	twirpMux := http.NewServeMux()
	twirpMux.Handle("/", twirpHandler)

	healthcheckPath := "/debug/running"
	healthcheckOutput := fmt.Sprintf("Running version %s in %s\n", gitCommit, env)

	healthcheckFunc := func(w http.ResponseWriter, r *http.Request) {
		_, err := w.Write([]byte(healthcheckOutput))
		if err != nil {
			logger.Error(fmt.Sprintf("Unable to write 'OK' to response for a "+healthcheckPath+" request: %v", err), nil, nil)
		}
	}

	twirpMux.HandleFunc(healthcheckPath, healthcheckFunc)

	tlsConfig, err := ephemeralcert.GenerateTLSConfig()
	if err != nil {
		logger.Fatal(fmt.Sprintf("Could not set up SSL Certificate: %v", err), nil, nil)
		os.Exit(1)
	}

	tlsBindAddress := fmt.Sprintf(":%d", config.Port)
	httpBindAddress := fmt.Sprintf(":%d", config.HTTPPort)

	tlsSocket, err := tls.Listen("tcp", tlsBindAddress, tlsConfig)
	if err != nil {
		logger.Fatal(fmt.Sprintf("Could not set up TLS socket: %v", err), nil, nil)
		os.Exit(1)
	}

	logger.Info(fmt.Sprintf("Server starting on %s,and %s", tlsBindAddress, httpBindAddress),
		map[string]interface{}{
			"ProcessAddressLaunchID": bs.ProcessIdentifier.LaunchID,
			"ProcessAddressMachine":  bs.ProcessIdentifier.Machine,
			"ProcessAddressVersion":  bs.ProcessIdentifier.Version,
			"ServiceTupleStage":      bs.ProcessIdentifier.Stage,
			"ServiceTupleSubstage":   bs.ProcessIdentifier.Substage,
			"ServiceTupleService":    bs.ProcessIdentifier.Service,
			"ServiceTupleRegion":     bs.ProcessIdentifier.Region,
		}, nil)

	//
	// kick off a go routine in which a separate http server can block on ListenAndServe using a mux with
	// the healthcheck endpoint.  This is the server that the LB Target Group will healthcheck,
	// because it only supports healthchecking over plain HTTP, not HTTPS, so it cannot use the healthcheck
	// endpoint that is configured on the twirpMux.  The healthcheck endpoint is configured on the twirpMux
	// because that is the endpoint that the healthcheck lambda, which is installed in the environment via
	// terraform, will get routed to when it requests /debug/running from the NLB itself.  The NLB will
	// forward that to the TLS port of the discovery server, and that goes to the twirpMux.  This is why
	// the healthcheck endpoint exists in two places, and why there are two servers running, and why one of
	// then must run in a gofunc.
	//
	go func() {
		healthcheckMux := http.NewServeMux()
		healthcheckMux.HandleFunc(healthcheckPath, healthcheckFunc)

		httpServer := &http.Server{
			Addr: httpBindAddress,
			Handler: healthcheckMux,
		}

		err = httpServer.ListenAndServe()
		if err != nil {
			logger.Fatal(fmt.Sprintf("ListenAndServe (healthcheck) failed with: %s", err), nil, nil)
			os.Exit(1)
		}
	}()

	server := &http.Server{
		Handler: twirpMux,
	}

	err = server.Serve(tlsSocket)

	if err != nil {
		logger.Fatal(fmt.Sprintf("error in ListenAndServe: %s", err), nil, nil)
		os.Exit(1)
	}
}

type DBConfigData struct {
	Username             string
	Password             string
	Engine               string
	Host                 string
	Port                 int
	DBName               string
	DBInstanceIdentifier string
}

func loadDBConfig(dbSecretName string) (*DBConfigData, error) {
	// Create a Secrets Manager client
	sess, err := session.NewSession()
	if err != nil {
		return nil, errors.Wrap(err, "Could not create new session")
	}
	svc := secretsmanager.New(sess)
	input := &secretsmanager.GetSecretValueInput{
		SecretId:     aws.String(dbSecretName),
		VersionStage: aws.String("AWSCURRENT"), // VersionStage defaults to AWSCURRENT if unspecified
	}

	// Retrieve config
	result, err := svc.GetSecretValue(input)
	if err != nil {
		return nil, errors.Wrap(err, "could not get secret value")
	}
	if result.SecretString == nil {
		return nil, errors.New("Got empty string for DB config for discoman postgres database")
	}

	// Parse response
	var data DBConfigData
	err = json.Unmarshal([]byte(*result.SecretString), &data)
	if err != nil {
		return nil, errors.Wrap(err, "Could not unmarshal secret string")
	}
	return &data, nil
}

func createDiscomanStore(config discoveryConfig, logger stacklogger.Logger) (model.Store, error) {
	dbconfig, err := loadDBConfig(config.DBSecretName)
	if err != nil {
		return nil, errors.WithMessage(err, "Unable to load DB config for discoman postgres database")
	}

	if config.DBHostOverride != "" {
		dbconfig.Host = config.DBHostOverride
	}

	db, err := postgres.New(dbconfig.Host, dbconfig.Port, dbconfig.Username, dbconfig.Password, dbconfig.DBName, "extensions_discovery", logger)
	if err != nil {
		return nil, errors.Wrap(err, "Unable to connect to discoman postgres database")
	}
	return db, nil
}
