package main

import (
	"context"
	"testing"

	"code.justin.tv/extensions/discovery/auth"
	"code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/discovery/data"
	"code.justin.tv/extensions/discovery/data/model/memory"
	"code.justin.tv/extensions/discovery/golibs/uuid"
	"code.justin.tv/extensions/discovery/manager"

	"github.com/stretchr/testify/assert"
)

func TestOrderCategories(t *testing.T) {
	t.Run("Must have a request", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		out, err := discoveryService.OrderCategories(ctx, nil)
		assert.Nil(t, out)
		assert.Equal(t, data.ErrMissingParameter, err)
	})

	t.Run("Must have a manager", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		out, err := discoveryService.OrderCategories(ctx, makeOrderCategoriesRequest([]string{"321"}))
		assert.Nil(t, out)
		assert.Equal(t, data.ErrUnavailable, err)
	})

	t.Run("Can reorder with a nonempty store", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		m := buildTestDiscoveryManager(store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, m)
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		cat1 := createCategoryWithNoErrors(t, discoveryService, m, validAddCategoryRequest())
		cat2 := createCategoryWithNoErrors(t, discoveryService, m, validAddCategoryRequest())
		cat3 := createCategoryWithNoErrors(t, discoveryService, m, validAddCategoryRequest())

		out, err := discoveryService.OrderCategories(ctx, makeOrderCategoriesRequest([]string{
			cat1.GetID(),
			cat2.GetID(),
			cat3.GetID(),
		}))
		assert.NotNil(t, out)
		assert.Nil(t, err)
	})

}

func makeOrderCategoriesRequest(categories []string) *discovery.OrderCategoriesRequest {
	return &discovery.OrderCategoriesRequest{
		Categories: categories,
	}
}
