package main

import (
	"context"
	"testing"

	"code.justin.tv/extensions/discovery/auth"
	"code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/discovery/data"
	"code.justin.tv/extensions/discovery/data/model/memory"
	"code.justin.tv/extensions/discovery/golibs/uuid"
	"code.justin.tv/extensions/discovery/manager"

	"github.com/stretchr/testify/assert"
)

func TestOrderFeaturedCarousel(t *testing.T) {
	t.Run("Must have a request", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		out, err := discoveryService.OrderCategories(ctx, nil)
		assert.Nil(t, out)
		assert.Equal(t, data.ErrMissingParameter, err)
	})

	t.Run("Must have a manager set", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		out, err := discoveryService.OrderFeaturedCarousel(ctx, makeOrderFeaturedCarouselRequest("123", []string{"1234"}))
		assert.Nil(t, out)
		assert.Equal(t, data.ErrUnavailable, err)
	})

	t.Run("Can reorder with a nonempty store", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		m := buildTestDiscoveryManager(store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, m)
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		carousel := createFeaturedCarousel(t, discoveryService, m, "123")
		entry1 := createFeaturedCarouselEntry(t, discoveryService, m, carousel.GetID())
		entry2 := createFeaturedCarouselEntry(t, discoveryService, m, carousel.GetID())
		entry3 := createFeaturedCarouselEntry(t, discoveryService, m, carousel.GetID())

		out, err := discoveryService.OrderFeaturedCarousel(ctx, makeOrderFeaturedCarouselRequest(carousel.GetID(), []string{
			entry2.GetID(),
			entry3.GetID(),
			entry1.GetID(),
		}))
		assert.NotNil(t, out)
		assert.Nil(t, err)
	})
}

func makeOrderFeaturedCarouselRequest(carouselID string, entryIDs []string) *discovery.OrderFeaturedCarouselRequest {
	return &discovery.OrderFeaturedCarouselRequest{
		CarouselID: carouselID,
		EntryIDs:   entryIDs,
	}
}
