package main

import (
	"context"
	"strings"
	"testing"

	"code.justin.tv/extensions/discovery/auth"
	"code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/discovery/data"
	"code.justin.tv/extensions/discovery/data/model/memory"
	"code.justin.tv/extensions/discovery/golibs/uuid"
	"code.justin.tv/extensions/discovery/manager"
	"code.justin.tv/extensions/discovery/twirputils"

	"github.com/stretchr/testify/assert"
)

func TestPutExtensionVersion(t *testing.T) {
	store := memory.New(uuid.NewSource())
	discoveryService := fakeServer(t, store)

	t.Run("Must have a request", func(t *testing.T) {
		ctx := context.Background()

		var req *discovery.PutExtensionVersionDiscoveryDataRequest

		out, err := discoveryService.PutExtensionVersionDiscoveryData(ctx, req)
		assert.Nil(t, out)
		assert.Equal(t, data.ErrMissingParameter, err)
	})

	t.Run("Must have an update", func(t *testing.T) {
		ctx := context.Background()

		req := &discovery.PutExtensionVersionDiscoveryDataRequest{}

		out, err := discoveryService.PutExtensionVersionDiscoveryData(ctx, req)
		assert.Nil(t, out)
		assert.Equal(t, data.ErrMissingParameter, err)
	})

	t.Run("Must have an extensionID", func(t *testing.T) {
		ctx := context.Background()

		req := &discovery.PutExtensionVersionDiscoveryDataRequest{
			UpdateData: &discovery.ExtensionVersionDiscoveryDataUpdate{},
		}

		out, err := discoveryService.PutExtensionVersionDiscoveryData(ctx, req)
		assert.Nil(t, out)
		assert.Equal(t, data.ErrMissingParameter, err)
	})

	t.Run("Must have a version", func(t *testing.T) {
		ctx := context.Background()

		req := &discovery.PutExtensionVersionDiscoveryDataRequest{
			UpdateData: &discovery.ExtensionVersionDiscoveryDataUpdate{
				ExtensionID: "testExtensionID",
			},
		}

		out, err := discoveryService.PutExtensionVersionDiscoveryData(ctx, req)
		assert.Nil(t, out)
		assert.Equal(t, data.ErrMissingParameter, err)
	})

	t.Run("Must have a store", func(t *testing.T) {
		ctx := context.Background()

		req := &discovery.PutExtensionVersionDiscoveryDataRequest{
			UpdateData: &discovery.ExtensionVersionDiscoveryDataUpdate{
				ExtensionID: "testExtensionID",
				Version:     "testExtensionVersion",
			},
		}

		out, err := discoveryService.PutExtensionVersionDiscoveryData(ctx, req)
		assert.Nil(t, out)
		assert.Equal(t, data.ErrUnavailable, err)
	})

	t.Run("Must have permissions", func(t *testing.T) {
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))

		req := &discovery.PutExtensionVersionDiscoveryDataRequest{
			UpdateData: &discovery.ExtensionVersionDiscoveryDataUpdate{
				ExtensionID: "testExtensionID",
				Version:     "testExtensionVersion",
			},
		}

		out, err := discoveryService.PutExtensionVersionDiscoveryData(ctx, req)
		assert.Nil(t, out)
		assert.Equal(t, data.ErrUnauthorized, err)
	})

	t.Run("Bad args", func(t *testing.T) {
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		t.Run("Name must be valid if updated", func(t *testing.T) {
			t.Run("empty", func(t *testing.T) {
				req := &discovery.PutExtensionVersionDiscoveryDataRequest{
					UpdateData: &discovery.ExtensionVersionDiscoveryDataUpdate{
						ExtensionID: "testExtensionID",
						Version:     "testExtensionVersion",
						Name:        twirputils.WrapString(""),
					},
				}

				out, err := discoveryService.PutExtensionVersionDiscoveryData(ctx, req)
				assert.Nil(t, out)
				assert.Equal(t, data.ErrNameLengthShort, err)
			})
			t.Run("too long", func(t *testing.T) {
				longName := makeLongString(data.MaxNameLength + 1)
				req := &discovery.PutExtensionVersionDiscoveryDataRequest{
					UpdateData: &discovery.ExtensionVersionDiscoveryDataUpdate{
						ExtensionID: "testExtensionID",
						Version:     "testExtensionVersion",
						Name:        twirputils.WrapString(longName),
					},
				}

				out, err := discoveryService.PutExtensionVersionDiscoveryData(ctx, req)
				assert.Nil(t, out)
				assert.Equal(t, data.ErrNameLengthLong, err)
			})
		})

		t.Run("Description must be valid if updated", func(t *testing.T) {
			longName := makeLongString(data.MaxDescriptionLength + 1)
			req := &discovery.PutExtensionVersionDiscoveryDataRequest{
				UpdateData: &discovery.ExtensionVersionDiscoveryDataUpdate{
					ExtensionID: "testExtensionID",
					Version:     "testExtensionVersion",
					Description: twirputils.WrapString(longName),
				},
			}

			out, err := discoveryService.PutExtensionVersionDiscoveryData(ctx, req)
			assert.Nil(t, out)
			assert.Equal(t, data.ErrDescriptionLength, err)
		})

		t.Run("Summary must be valid if updated", func(t *testing.T) {
			longName := makeLongString(data.MaxSummaryLength + 1)
			req := &discovery.PutExtensionVersionDiscoveryDataRequest{
				UpdateData: &discovery.ExtensionVersionDiscoveryDataUpdate{
					ExtensionID: "testExtensionID",
					Version:     "testExtensionVersion",
					Summary:     twirputils.WrapString(longName),
				},
			}

			out, err := discoveryService.PutExtensionVersionDiscoveryData(ctx, req)
			assert.Nil(t, out)
			assert.Equal(t, data.ErrSummaryLength, err)
		})

		t.Run("Support email must be valid if updated", func(t *testing.T) {
			t.Run("empty", func(t *testing.T) {
				req := &discovery.PutExtensionVersionDiscoveryDataRequest{
					UpdateData: &discovery.ExtensionVersionDiscoveryDataUpdate{
						ExtensionID:  "testExtensionID",
						Version:      "testExtensionVersion",
						SupportEmail: twirputils.WrapString(""),
					},
				}

				out, err := discoveryService.PutExtensionVersionDiscoveryData(ctx, req)
				assert.Nil(t, out)
				assert.Equal(t, data.ErrInvalidSupportEmail, err)
			})
			t.Run("not email 1", func(t *testing.T) {
				req := &discovery.PutExtensionVersionDiscoveryDataRequest{
					UpdateData: &discovery.ExtensionVersionDiscoveryDataUpdate{
						ExtensionID:  "testExtensionID",
						Version:      "testExtensionVersion",
						SupportEmail: twirputils.WrapString("notemail"),
					},
				}

				out, err := discoveryService.PutExtensionVersionDiscoveryData(ctx, req)
				assert.Nil(t, out)
				assert.Equal(t, data.ErrInvalidSupportEmail, err)
			})
			t.Run("not email 2", func(t *testing.T) {
				req := &discovery.PutExtensionVersionDiscoveryDataRequest{
					UpdateData: &discovery.ExtensionVersionDiscoveryDataUpdate{
						ExtensionID:  "testExtensionID",
						Version:      "testExtensionVersion",
						SupportEmail: twirputils.WrapString("notemail@nothe@2.com"),
					},
				}

				out, err := discoveryService.PutExtensionVersionDiscoveryData(ctx, req)
				assert.Nil(t, out)
				assert.Equal(t, data.ErrInvalidSupportEmail, err)
			})
		})

		t.Run("EULATOS URL must be valid if updated", func(t *testing.T) {
			t.Run("empty", func(t *testing.T) {
				req := &discovery.PutExtensionVersionDiscoveryDataRequest{
					UpdateData: &discovery.ExtensionVersionDiscoveryDataUpdate{
						ExtensionID: "testExtensionID",
						Version:     "testExtensionVersion",
						EULATOSURL:  twirputils.WrapString(""),
					},
				}

				out, err := discoveryService.PutExtensionVersionDiscoveryData(ctx, req)
				assert.Nil(t, out)
				assert.Equal(t, data.ErrInvalidTermsURL, err)
			})
			t.Run("not url", func(t *testing.T) {
				req := &discovery.PutExtensionVersionDiscoveryDataRequest{
					UpdateData: &discovery.ExtensionVersionDiscoveryDataUpdate{
						ExtensionID: "testExtensionID",
						Version:     "testExtensionVersion",
						EULATOSURL:  twirputils.WrapString("noturl"),
					},
				}

				out, err := discoveryService.PutExtensionVersionDiscoveryData(ctx, req)
				assert.Nil(t, out)
				assert.Equal(t, data.ErrInvalidTermsURL, err)
			})
			t.Run("wrong scheme", func(t *testing.T) {
				req := &discovery.PutExtensionVersionDiscoveryDataRequest{
					UpdateData: &discovery.ExtensionVersionDiscoveryDataUpdate{
						ExtensionID: "testExtensionID",
						Version:     "testExtensionVersion",
						EULATOSURL:  twirputils.WrapString("ftp://example.com"),
					},
				}

				out, err := discoveryService.PutExtensionVersionDiscoveryData(ctx, req)
				assert.Nil(t, out)
				assert.Equal(t, data.ErrInvalidTermsURL, err)
			})
		})

		t.Run("Privacy policy URL must be valid if updated", func(t *testing.T) {
			t.Run("empty", func(t *testing.T) {
				req := &discovery.PutExtensionVersionDiscoveryDataRequest{
					UpdateData: &discovery.ExtensionVersionDiscoveryDataUpdate{
						ExtensionID:      "testExtensionID",
						Version:          "testExtensionVersion",
						PrivacyPolicyURL: twirputils.WrapString(""),
					},
				}

				out, err := discoveryService.PutExtensionVersionDiscoveryData(ctx, req)
				assert.Nil(t, out)
				assert.Equal(t, data.ErrInvalidPrivacyURL, err)
			})
			t.Run("not url", func(t *testing.T) {
				req := &discovery.PutExtensionVersionDiscoveryDataRequest{
					UpdateData: &discovery.ExtensionVersionDiscoveryDataUpdate{
						ExtensionID:      "testExtensionID",
						Version:          "testExtensionVersion",
						PrivacyPolicyURL: twirputils.WrapString("noturl"),
					},
				}

				out, err := discoveryService.PutExtensionVersionDiscoveryData(ctx, req)
				assert.Nil(t, out)
				assert.Equal(t, data.ErrInvalidPrivacyURL, err)
			})
			t.Run("wrong scheme", func(t *testing.T) {
				req := &discovery.PutExtensionVersionDiscoveryDataRequest{
					UpdateData: &discovery.ExtensionVersionDiscoveryDataUpdate{
						ExtensionID:      "testExtensionID",
						Version:          "testExtensionVersion",
						PrivacyPolicyURL: twirputils.WrapString("ftp://example.com"),
					},
				}

				out, err := discoveryService.PutExtensionVersionDiscoveryData(ctx, req)
				assert.Nil(t, out)
				assert.Equal(t, data.ErrInvalidPrivacyURL, err)
			})
		})

		t.Run("Author name must be valid if updated", func(t *testing.T) {
			longName := makeLongString(data.MaxAuthorNameLength + 1)
			req := &discovery.PutExtensionVersionDiscoveryDataRequest{
				UpdateData: &discovery.ExtensionVersionDiscoveryDataUpdate{
					ExtensionID: "testExtensionID",
					Version:     "testExtensionVersion",
					AuthorName:  twirputils.WrapString(longName),
				},
			}

			out, err := discoveryService.PutExtensionVersionDiscoveryData(ctx, req)
			assert.Nil(t, out)
			assert.Equal(t, data.ErrAuthorNameLength, err)
		})
	})

	t.Run("Can create a version", func(t *testing.T) {
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		req := &discovery.PutExtensionVersionDiscoveryDataRequest{
			UpdateData: &discovery.ExtensionVersionDiscoveryDataUpdate{
				ExtensionID:      "testExtensionID",
				Version:          "testExtensionVersion",
				Name:             twirputils.WrapString("Cool name"),
				Description:      twirputils.WrapString("Cool description"),
				Summary:          twirputils.WrapString("Cool summary"),
				SupportEmail:     twirputils.WrapString("support@example.com"),
				EULATOSURL:       twirputils.WrapString("https://example.com/eulatos.html"),
				PrivacyPolicyURL: twirputils.WrapString("https://example.com/privacy.html"),
				AuthorName:       twirputils.WrapString("Cool author"),
			},
		}

		out, err := discoveryService.PutExtensionVersionDiscoveryData(ctx, req)
		assert.NotNil(t, out)
		assert.Nil(t, err)
	})
}

// makeLongString makes a string of the provided length
func makeLongString(length int) string {
	return strings.Repeat("%", length)
}
