package main

import (
	"context"
	"testing"

	"code.justin.tv/extensions/discovery/auth"
	discovery "code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/discovery/data"
	"code.justin.tv/extensions/discovery/data/model"
	"code.justin.tv/extensions/discovery/data/model/memory"
	"code.justin.tv/extensions/discovery/golibs/uuid"
	"code.justin.tv/extensions/discovery/manager"
	"code.justin.tv/extensions/discovery/twirputils"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestRemoveExtensionFromCategory(t *testing.T) {
	t.Run("must have request", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)

		ctx := context.Background()
		out, err := discoveryService.RemoveExtensionFromCategory(ctx, nil)
		assert.Nil(t, out)
		assert.Equal(t, data.ErrMissingParameter, err)
	})

	t.Run("must have manager", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)

		ctx := auth.Store(context.Background(), &auth.FakeCredentials{Role: auth.Categorizer})

		out, err := discoveryService.RemoveExtensionFromCategory(ctx, makeRemoveExtensionFromCategoryRequest("someext", "somecat"))
		assert.Nil(t, out)
		assert.Equal(t, data.ErrUnavailable, err)
	})

	t.Run("category must exist", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)

		ctx := manager.StoreDiscoveryManager(context.Background(), buildTestDiscoveryManager(store))
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		out, err := discoveryService.RemoveExtensionFromCategory(ctx, makeRemoveExtensionFromCategoryRequest("someext", "somecat"))
		assert.Nil(t, out)
		assert.Equal(t, data.ErrNotFound, err)
	})

	t.Run("must be authorized", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)

		ctx := manager.StoreDiscoveryManager(context.Background(), buildTestDiscoveryManager(store))
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.NoRole})

		cat, err := store.CreateCategory(&discovery.AddCategoryRequest{
			Category: &discovery.CategoryUpsert{
				Type:    twirputils.WrapString(model.CategoryTypeDeveloper),
				SortKey: twirputils.WrapString(model.CategorySortkeyManual),
				Visible: twirputils.WrapBool(true),
			},
		})
		require.NoError(t, err)

		out, err := discoveryService.RemoveExtensionFromCategory(ctx, makeRemoveExtensionFromCategoryRequest("someext", cat.ID))
		assert.Nil(t, out)
		assert.Equal(t, data.ErrUnauthorized, err)
	})

	t.Run("must not be pseudo category", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)

		ctx := manager.StoreDiscoveryManager(context.Background(), buildTestDiscoveryManager(store))
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		cat, err := store.CreateCategory(&discovery.AddCategoryRequest{
			Category: &discovery.CategoryUpsert{
				Type:     twirputils.WrapString(model.CategoryTypePseudo),
				SortKey:  twirputils.WrapString(model.CategorySortkeyManual),
				Visible:  twirputils.WrapBool(false),
				Readonly: twirputils.WrapBool(true),
				Order:    twirputils.WrapFloat(0),
				Slug:     twirputils.WrapString("aslug"),
			},
		})
		require.NoError(t, err)

		out, err := discoveryService.RemoveExtensionFromCategory(ctx, makeRemoveExtensionFromCategoryRequest("someext", cat.ID))
		assert.Nil(t, out)
		assert.Equal(t, data.ErrInvalidCategoryUpdate, err)
	})

	t.Run("success", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := manager.StoreDiscoveryManager(context.Background(), buildTestDiscoveryManager(store))
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Seer | auth.Categorizer})

		cat, err := store.CreateCategory(&discovery.AddCategoryRequest{
			Category: &discovery.CategoryUpsert{
				Type:    twirputils.WrapString(model.CategoryTypeCurated),
				SortKey: twirputils.WrapString(model.CategorySortkeyManual),
			},
		})
		require.NoError(t, err)

		req := makeAddExtensionToCategoryRequest("testExtensionID", cat.ID, "testVersionID", nil)
		addResp, err := discoveryService.AddExtensionToCategory(ctx, req)
		require.Nil(t, err)
		require.NotNil(t, addResp)

		removeReq := makeRemoveExtensionFromCategoryRequest(req.GetExtensionID(), cat.ID)
		removeResp, err := discoveryService.RemoveExtensionFromCategory(ctx, removeReq)
		require.Nil(t, err)
		require.NotNil(t, removeResp.GetExtensionCategoryMembership())
		assert.Equal(t, cat.ID, removeResp.GetExtensionCategoryMembership().GetCategoryID())
	})
}

func makeRemoveExtensionFromCategoryRequest(extensionID, categoryID string) *discovery.RemoveExtensionFromCategoryRequest {
	return &discovery.RemoveExtensionFromCategoryRequest{
		CategoryID:  categoryID,
		ExtensionID: extensionID,
	}
}
