package main

import (
	"context"
	"testing"

	"code.justin.tv/extensions/discovery/auth"
	"code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/discovery/data"
	"code.justin.tv/extensions/discovery/data/model"
	"code.justin.tv/extensions/discovery/data/model/memory"
	"code.justin.tv/extensions/discovery/golibs/uuid"
	"code.justin.tv/extensions/discovery/manager"
	"code.justin.tv/extensions/discovery/twirputils"

	"github.com/stretchr/testify/assert"
)

func TestUpdateCategory(t *testing.T) {
	t.Run("Must have a request", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		out, err := discoveryService.UpdateCategory(ctx, nil)
		assert.Nil(t, out)
		assert.Equal(t, data.ErrMissingParameter, err)
	})

	t.Run("Must have a manager", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		out, err := discoveryService.UpdateCategory(ctx, makeUpdateCategoryRequest("123", nil))
		assert.Nil(t, out)
		assert.Equal(t, data.ErrUnavailable, err)
	})

	t.Run("Must be allowed to edit categories", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))

		out, err := discoveryService.UpdateCategory(ctx, makeUpdateCategoryRequest("123", nil))
		assert.Nil(t, out)
		assert.Equal(t, data.ErrUnauthorized, err)
	})

	t.Run("Category type must make sense", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		out, err := discoveryService.UpdateCategory(ctx, makeUpdateCategoryRequest("123", &discovery.CategoryUpsert{
			Type: twirputils.WrapString("nonsense"),
		}))
		assert.Nil(t, out)
		assert.Equal(t, data.ErrUnknownCategoryType, err)
	})

	t.Run("Category sort key must make sense", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		out, err := discoveryService.UpdateCategory(ctx, makeUpdateCategoryRequest("123", &discovery.CategoryUpsert{
			SortKey: twirputils.WrapString("nonsense"),
		}))
		assert.Nil(t, out)
		assert.Equal(t, data.ErrUnknownCategorySortKey, err)
	})

	t.Run("Can update a category successfully", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		m := buildTestDiscoveryManager(store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, m)
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		cat := createCategoryWithNoErrors(t, discoveryService, m, validAddCategoryRequest())

		out, err := discoveryService.UpdateCategory(ctx, makeUpdateCategoryRequest(cat.GetID(), &discovery.CategoryUpsert{
			SortKey: twirputils.WrapString("popularity"),
			Type:    twirputils.WrapString(model.CategoryTypeDeveloper),
		}))
		assert.NotNil(t, out)
		assert.Nil(t, err)
	})
}

func makeUpdateCategoryRequest(categoryID string, cat *discovery.CategoryUpsert) *discovery.UpdateCategoryRequest {
	return &discovery.UpdateCategoryRequest{
		CategoryID: categoryID,
		Category:   cat,
	}
}
