package main

import (
	"context"
	"testing"

	"code.justin.tv/extensions/discovery/auth"
	"code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/discovery/data"
	"code.justin.tv/extensions/discovery/data/model/memory"
	"code.justin.tv/extensions/discovery/golibs/uuid"
	"code.justin.tv/extensions/discovery/manager"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestUpdateFeaturedSchedule(t *testing.T) {
	t.Run("Must have a request", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		out, err := discoveryService.UpdateFeaturedSchedule(ctx, nil)
		assert.Nil(t, out)
		assert.Equal(t, data.ErrMissingParameter, err)
	})

	t.Run("Must have a manager", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		out, err := discoveryService.UpdateFeaturedSchedule(ctx, makeUpdateFeaturedScheduleRequest("123", "description"))
		assert.Nil(t, out)
		assert.Equal(t, data.ErrUnavailable, err)
	})

	t.Run("Must be allowed to edit featured schedules", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, buildTestDiscoveryManager(store))

		out, err := discoveryService.UpdateFeaturedSchedule(ctx, makeUpdateFeaturedScheduleRequest("123", "description"))
		assert.Nil(t, out)
		assert.Equal(t, data.ErrUnauthorized, err)
	})

	t.Run("Can update a schedule successfully", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		discoveryService := fakeServer(t, store)
		m := buildTestDiscoveryManager(store)
		ctx := context.Background()
		ctx = manager.StoreDiscoveryManager(ctx, m)
		ctx = auth.Store(ctx, &auth.FakeCredentials{Role: auth.Categorizer})

		schedule, err := discoveryService.AddFeaturedSchedule(ctx, makeAddFeaturedScheduleRequest("description"))
		require.NotNil(t, schedule)
		assert.Nil(t, err)

		carousel := createFeaturedCarousel(t, discoveryService, m, schedule.GetFeaturedSchedule().GetID())
		_ = createFeaturedCarouselEntry(t, discoveryService, m, carousel.GetID())

		out, err := discoveryService.UpdateFeaturedSchedule(ctx, makeUpdateFeaturedScheduleRequest(schedule.GetFeaturedSchedule().GetID(), "newdescription"))
		assert.NotNil(t, out)
		assert.Nil(t, err)
	})
}

func makeUpdateFeaturedScheduleRequest(scheduleID, description string) *discovery.UpdateFeaturedScheduleRequest {
	return &discovery.UpdateFeaturedScheduleRequest{
		ScheduleID: scheduleID,
		FeaturedSchedule: &discovery.FeaturedScheduleUpsert{
			Description: description,
		},
	}
}
