package main

import (
	"context"
	"testing"

	"code.justin.tv/extensions/discovery/auth"
	"code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/discovery/data/model"
	"code.justin.tv/extensions/discovery/interfaces"
	"code.justin.tv/extensions/discovery/manager"
	"code.justin.tv/extensions/discovery/twirputils"

	"github.com/stretchr/testify/require"
)

/*
 * This file holds utilities used in tests rather than testing utils.go
 */

func fakeServer(t *testing.T, store model.Store) *discoveryServer {
	return &discoveryServer{
		store: store,
	}
}

func buildTestDiscoveryManager(store model.Store) interfaces.DiscoveryManager {
	return manager.New(store)
}

func createFeaturedSchedule(t *testing.T, discoveryService discovery.Discovery, m interfaces.DiscoveryManager) *discovery.FeaturedScheduleDocument {
	creds := &auth.FakeCredentials{Role: auth.Categorizer}
	ctx := auth.Store(context.Background(), creds)
	ctx = manager.StoreDiscoveryManager(ctx, m)
	schedule, err := discoveryService.AddFeaturedSchedule(ctx, makeAddFeaturedScheduleRequest("description"))

	require.NotNil(t, schedule)
	require.Nil(t, err)
	return schedule.GetFeaturedSchedule()
}

func createFeaturedCarousel(t *testing.T, discoveryService discovery.Discovery, m interfaces.DiscoveryManager, scheduleID string) *discovery.FeaturedCarouselDocument {
	creds := &auth.FakeCredentials{Role: auth.Categorizer}
	ctx := auth.Store(context.Background(), creds)
	ctx = manager.StoreDiscoveryManager(ctx, m)
	carousel, err := discoveryService.AddFeaturedCarousel(ctx, makeAddFeaturedCarouselRequest(t, scheduleID))

	require.NotNil(t, carousel)
	require.Nil(t, err)
	return carousel.GetFeaturedCarousel()
}

func createFeaturedCarouselEntry(t *testing.T, discoveryService discovery.Discovery, m interfaces.DiscoveryManager, carouselID string) *discovery.FeaturedCarouselEntryDocument {
	creds := &auth.FakeCredentials{Role: auth.Categorizer}
	ctx := auth.Store(context.Background(), creds)
	ctx = manager.StoreDiscoveryManager(ctx, m)
	entry, err := discoveryService.AddFeaturedCarouselEntry(ctx, makeAddFeaturedCarouselEntryRequest(carouselID))

	require.NotNil(t, entry)
	require.Nil(t, err)
	return entry.GetFeaturedCarouselEntry()
}

func createCategoryWithNoErrors(t *testing.T, d discovery.Discovery, m interfaces.DiscoveryManager, req *discovery.AddCategoryRequest) *discovery.CategoryDocument {
	creds := &auth.FakeCredentials{Role: auth.Categorizer}
	ctx := auth.Store(context.Background(), creds)
	ctx = manager.StoreDiscoveryManager(ctx, m)
	cat, err := d.AddCategory(ctx, req)

	require.NoError(t, err)
	require.NotNil(t, cat)
	return cat.Category
}

func validAddCategoryRequest() *discovery.AddCategoryRequest {
	return &discovery.AddCategoryRequest{
		Category: &discovery.CategoryUpsert{
			Order:    twirputils.WrapFloat(2),
			Readonly: twirputils.WrapBool(false),
			SortKey:  twirputils.WrapStringPointer(&model.CategorySortkeyManual),
			Type:     twirputils.WrapStringPointer(&model.CategoryTypeDeveloper),
			Visible:  twirputils.WrapBool(true),
		},
	}
}
