package data

import (
	"net/http"

	"fmt"

	"github.com/jixwanwang/apiutils"
	"github.com/twitchtv/twirp"
)

var (
	// ErrCategoryNotFound indicates that the requested category was not present in the database
	ErrCategoryNotFound = twirp.NewError(twirp.NotFound, "Category not found")
	// ErrConcurrency is returned by a store when save requests collide
	ErrConcurrency = twirp.NewError(twirp.AlreadyExists, "Concurrency violation")
	// ErrExtensionNotInCategory indicates that we tried to remove an extension from a category but it wasn't in there to begin with
	ErrExtensionNotInCategory = twirp.NewError(twirp.NotFound, "Extension not in category")
	// ErrInvalidCategoryUpdate indicates that the requested change to a category was not permitted
	ErrInvalidCategoryUpdate = twirp.NewError(twirp.InvalidArgument, "Illegal category update")
	// ErrInvalidPagination indicates the passed pagination options were invalid
	ErrInvalidPagination = twirp.NewError(twirp.InvalidArgument, "Pagination parameters are invalid")
	// ErrInvalidGameID indicates the passed game id was invalid
	ErrInvalidGameID = twirp.NewError(twirp.InvalidArgument, "Game id is invalid")
	// ErrGameNotFound indicates that the passed game id was not found on the extension
	// TODO EMS used 422 for this.  Is 400 OK?
	ErrGameNotFound = twirp.NewError(twirp.InvalidArgument, "Game id not found on extension")
	// ErrMissingParameter indicates that a required parameter was missing
	ErrMissingParameter = twirp.NewError(twirp.InvalidArgument, "Missing required parameter")
	// ErrNotFound is returned by a store when a load request can't be fulfilled
	ErrNotFound = twirp.NewError(twirp.NotFound, "The record was not found")
	// ErrNotImplemented is a convenience error so things can conform to an interface before they're done
	ErrNotImplemented = twirp.NewError(twirp.Unimplemented, "Not implemented yet")
	// ErrRequiresSlug is returned if a pseudo category is created without a slug
	ErrRequiresSlug = twirp.NewError(twirp.InvalidArgument, "Slug is required for pseudo categories")
	// ErrSlugTaken indicates a category with the provided slug already exists
	ErrSlugTaken = twirp.NewError(twirp.AlreadyExists, "A category with that slug already exists")
	// ErrTooManyRequests is thrown if a store is under too much request pressure and requires a cooldown period
	// TODO twirp doesn't have a 429
	ErrTooManyRequests = apiutils.NewErrorResponse(http.StatusTooManyRequests, "Too many requests")
	// ErrTranslationCountMismatch means that the document builder was given the wrong number of translations
	ErrTranslationCountMismatch = twirp.NewError(twirp.Internal, "Wrong number of category translations given")
	// ErrUnauthorized indicates the user is attempting to perform an action they're not allowed to
	ErrUnauthorized = twirp.NewError(twirp.Unauthenticated, "Unauthorized")
	// ErrUnavailable indicates that a context temporarily doesn't support functionality
	ErrUnavailable = twirp.NewError(twirp.Unknown, "Service not available")
	// ErrUnknownCategorySortKey indicates that an unknown category sort key was attempted to be created
	ErrUnknownCategorySortKey = twirp.NewError(twirp.InvalidArgument, "Unknown category sort key")
	// ErrUnknownCategoryType indicates that an unknown category type was attempted to be created
	ErrUnknownCategoryType = twirp.NewError(twirp.InvalidArgument, "Unknown category type")
	// ErrMissingEntryContent indicates that a featured carousel entry was created without any content
	ErrMissingEntryContent = twirp.NewError(twirp.InvalidArgument, "missing 'content' field")
	// ErrMissingEntryCarouselID indicates that a featured carousel entry was created without any carousel ID
	ErrMissingEntryCarouselID = twirp.NewError(twirp.InvalidArgument, "missing 'carousel_id' field")
	// ErrLogicViolation means that there was an internal failure resulting in the violation of assumptions/preconditions
	ErrLogicViolation = twirp.NewError(twirp.Internal, "Internal failure caused assumption violation")
	// ErrNotReadOnly indicates that the category to be deleted isn't in read only mode
	ErrNotReadOnly = twirp.NewError(twirp.InvalidArgument, "Category is not readonly")
	// ErrCategoryVisible indicates that the category to be deleted is still listable
	ErrCategoryVisible = twirp.NewError(twirp.InvalidArgument, "Category is still visible")
	// ErrNonEmptyCategory indicates that the category to be deleted still has associated extensions
	ErrNonEmptyCategory = twirp.NewError(twirp.InvalidArgument, "Category still has extensions associated with it")
	// ErrNameLengthLong indicates that the provided name is too long
	ErrNameLengthLong = twirp.NewError(twirp.InvalidArgument, fmt.Sprintf("Extension names are limited to %d characters", MaxNameLength))
	// ErrNameLengthShort indicates that the provided name is too short
	ErrNameLengthShort = twirp.NewError(twirp.InvalidArgument, "Extension name cannot be empty")
	// ErrDescriptionLength indicates that the provided description is too long
	ErrDescriptionLength = twirp.NewError(twirp.InvalidArgument, fmt.Sprintf("Extension descriptions are limited to %d characters", MaxDescriptionLength))
	// ErrNonEmptyCategory indicates that the category to be deleted still has associated extensions
	ErrSummaryLength = twirp.NewError(twirp.InvalidArgument, fmt.Sprintf("Extension summaries are limited to %d characters", MaxSummaryLength))
	// ErrInvalidSupportEmail indicates that the support email is not valid
	ErrInvalidSupportEmail = twirp.NewError(twirp.InvalidArgument, "Support email is not valid")
	// ErrInvalidTermsURL indicates that the EULA/TOS URL is not valid
	ErrInvalidTermsURL = twirp.NewError(twirp.InvalidArgument, "EULA/TOS URL is not valid")
	// ErrInvalidPrivacyURL indicates that the privacy policy URL is not valid
	ErrInvalidPrivacyURL = twirp.NewError(twirp.InvalidArgument, "Privacy policy URL is not valid")
	// ErrAuthorNameLength indicates that the provided author name is too long
	ErrAuthorNameLength = twirp.NewError(twirp.InvalidArgument, fmt.Sprintf("Author name must be fewer than %d characters", MaxAuthorNameLength))
)
