# Discovery Migrations
In order to support migrations, we're using a relatively simple library called
[Gormigrate](https://github.com/go-gormigrate/gormigrate). The primary reasoning
for this is because it supports transactions which should allow us to feel safe
running migrations at system startup (multiple servers running the same migration
not inside a transaction is scary).

This library is also in charge of setting up the database for us the first time.
Documentation around this can be found [here](https://github.com/go-gormigrate/gormigrate#having-a-separated-function-for-initializing-the-schema).
And our implementation can be found [here](../db.go).

# Setting up a Migration
There are currently no migrations because the project has just started but the idea
around setting up a migration is to simple add a new file to this directory with
the name of the migration.

Example:

Filename: `discoman/data/migrations/categories_create_column_slug.go`
```golang
package migrations

import (
	gormigrate "gopkg.in/gormigrate.v1"

	"code.justin.tv/gds/gds/extensions/ems/discoman/data/model"
	"github.com/jinzhu/gorm"
)

// CategoriesCreateColumnSlug adds a new column to the categories table to hold a "slug"
// which is a human-readable string used in URLs linking directly to a category page.
var CategoriesCreateColumnSlug = &gormigrate.Migration{
	ID: "201804261600",
	Migrate: func(tx *gorm.DB) error {
		type Category struct {
			Slug string
		}

		err := tx.AutoMigrate(&Category{}).Error
		if err != nil {
			return err
		}

		return tx.Model(model.Category{}).Error
	},
	Rollback: func(tx *gorm.DB) error {
		return tx.DropColumn("slug").Error
	},
}
```

You would then add the reference to this migration to the list of migrations
in [discoman/data/model/postgres/store.go](../model/postgres/store.go).

```golang
m := gormigrate.New(db, &gormigrate.Options{
	TableName:      "migrations",
	IDColumnName:   "id",
	IDColumnSize:   255,
	UseTransaction: true,
}, []*gormigrate.Migration{
	migrations.CategoriesCreateColumnSlug,
})
```

### Important things to note
1. Keep in mind that migrations will run in an transaction which requires always
checking against/returning `.Error`.

2. The ID field for migrations must be unique. This should be easily achievable by
following the format of YYYYMMDDHHMM for our IDs.

3. [Gormigrate](https://github.com/go-gormigrate/gormigrate#usage) recommends you
duplicate the model in your migration. So be sure to use the exact name of the
`type` used in the model directory. You don't need to duplicate all the fields,
just the ones you want to add. Refer to `gormigrate`'s documentation for more
assistance.
