package memory

import (
	"sort"

	"code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/discovery/data/model"
)

func (s *store) GetCategories(req *discovery.GetCategoriesRequest) ([]*model.Category, []*model.CategoryTranslation, int32, error) {
	defer s.mutex.Unlock()
	s.mutex.Lock()

	languageCode, err := s.getOrCreateLanguageCode(req.GetLanguage())
	if err != nil {
		return nil, nil, 0, err
	}

	defaultLanguageCode, err := s.getOrCreateLanguageCode(model.DefaultCategoryLanguageCode)
	if err != nil {
		return nil, nil, 0, err
	}

	allCategories := []*model.Category{}
	for _, val := range s.categories {
		if req.GetType() != "" && val.Type != req.GetType() {
			continue
		}
		if !req.GetIncludeHidden() && !val.Visible {
			continue
		}
		if !req.GetIncludeDeleted() && val.Deleted {
			continue
		}
		allCategories = append(allCategories, val)
	}

	numCategories := len(allCategories)

	offset := int(req.GetOffset())
	limit := int(req.GetLimit())

	if numCategories == 0 || offset >= numCategories {
		return []*model.Category{}, []*model.CategoryTranslation{}, int32(numCategories), nil
	}

	sort.Slice(allCategories, func(i, j int) bool {
		return allCategories[i].Order < allCategories[j].Order
	})
	upper := offset + limit
	if upper > numCategories {
		upper = numCategories
	}

	cats := allCategories[offset:upper]
	translations := make([]*model.CategoryTranslation, 0, len(cats))
	for _, cat := range cats {
		var translation *model.CategoryTranslation
		translationMap, ok := s.translations[cat.ID]
		if !ok {
			// no translation for this category has *ever* been made.
			translation = nil
		} else {
			translation, ok = translationMap[languageCode.ID]
			if !ok {
				// no translation in the requested language, try english
				translation, ok = translationMap[defaultLanguageCode.ID]
				if !ok {
					// no english translation.
					translation = nil
				}
			}
		}

		translations = append(translations, translation)
	}

	return cats, translations, int32(numCategories), nil
}
